! TQ library example to illustrate the use the adaptive interpolation
! scheme.
!
! This example calculates the liquidus temperature in a part of the
! C-CR-FE system and displays a selection of the results.
!
module mtqex15

  integer :: TC_NWSG,TC_NWSE,TC_NWSW,TC_STRLEN_PHASES,TC_STRLEN_MAX
  parameter (TC_NWSG=40000  ,TC_NWSE=50000,TC_NWSW=40000)
  parameter (TC_STRLEN_PHASES=24,TC_STRLEN_MAX=256)

  integer, allocatable, public, save ::   iwsg(:),iwse(:)
  double precision, allocatable, public, save :: arr(:)
  integer, public,save :: nel,ibranch

end module mtqex15

subroutine tqex15_init(liquid_name)
  use mtqex15
  implicit none
  character*(*) :: liquid_name

  integer :: i,nph,ill,ierr,imb,iph_liquid
  logical :: sg1err
  logical, allocatable :: idepel(:)
  integer, allocatable :: iphsta(:)
  character :: str*(TC_STRLEN_MAX)
  character*(TC_STRLEN_PHASES), allocatable :: phasestr(:)
  logical :: tiscnd,tiscst,piscnd,piscst
  integer :: idisct,logstep
  double precision :: t,tmin,tmax,p,pmin,pmax,tmemfrac
  double precision, allocatable :: xmin(:),xmax(:),phamnt(:)


  ! get the number of components
  call tqgnc(nel,iwsg,iwse)
  if (sg1err(ierr)) return

  ! get the number of phases
  call tqgnp(nph,iwsg,iwse)
  if (sg1err(ierr)) return

  ! allocate arrays needed for the calculation
  allocate(idepel(nel))
  allocate(xmin(nel))
  allocate(xmax(nel))
  allocate(arr(nel+2))
  allocate(phasestr(nph))
  allocate(iphsta(nph))
  allocate(phamnt(nph))


  ! find the index of the liquid phase and append '#1' to all phases
  ill=len_trim(liquid_name)
  do i=1,nph
     call tqgpn(i,phasestr(i),iwsg,iwse)
     if (index(phasestr(i),'#').eq.0) then
        phasestr(i)(len_trim(phasestr(i))+1:)='#1'
     endif
     if (sg1err(ierr)) return
     if (liquid_name(1:ill).eq.phasestr(i)(1:ill)) then
        iph_liquid=i
     endif
  enddo

  ! initialize the interpolation scheme
  ierr=0
  call tqips_init_top(ierr,iwsg,iwse)

  ! define that temperature varies in this calculation
  ! (is neither constant nor a condition)
  tiscnd=.FALSE.
  tiscst=.FALSE.
  ! define that pressure is fixed in this calculation
  ! (is a condition and do not vary)
  piscnd=.TRUE.
  piscst=.TRUE.

  ! define that all compositions are given
  ! (the content of all components are conditions)
  do i=1,nel
     idepel(i)=.TRUE.
  enddo

  ! use a linear discretization in composition space
  idisct=1
  ! use a virtual spacing corresponding to 100 points in composition 
  ! and temperature space
  logstep=2
  ! assume a minimum, maximum and start value for temperature
  tmin=1800.0d0
  tmax=2000.0d0
  t=(tmin+tmax)*0.5d0
  ! assume a minimum, maximum and start value for pressure
  p=101325.0d0
  pmin=101325.0d0
  pmax=101325.0d0
  ! allow the interpolation scheme to use a maximum of 70% of the 
  ! free memory
  tmemfrac=0.7d0


  ! retrieve the amount of free memory
  call tqgfmm(imb)
  ! set the maximum memory allocation to be 100Mb on a 32bit machine
  if (sizeof(imb) .eq. 4 ) then
     tmemfrac=min(tmemfrac,100.d0/dble(imb))
  endif

  ! set the minimum and maximum range of the composition of the
  ! components
  do i=1,nel
     xmin(i)=0.d0
     xmax(i)=1.d0
  enddo

  ! set all phases ENTERED except the liquid phase, which is to
  ! be FIXED with amount 1
  do i=1,nph
     iphsta(i)=1
     phamnt(i)=0.d0
  enddo
  iphsta(iph_liquid)=4
  phamnt(iph_liquid)=1.0D0

  ! initialize this branch in the interpolation scheme
  ierr=0
  call tqips_init_branch(tiscnd,tiscst,piscnd,piscst, &
       idepel, &
       idisct,logstep, &
       iphsta, &
       t,tmin,tmax,p,pmin,pmax,tmemfrac, &
       phamnt,xmin,xmax, &
       ibranch,ierr, &
       iwsg,iwse)



  ! initialize a function to retrieve the temperature
  str='T'
  ierr=0
  call tqips_init_function(str,ibranch,ierr,iwsg,iwse)


  ! deallocate temporary variables needed for the setup
  deallocate(idepel)
  deallocate(xmin)
  deallocate(xmax)
  deallocate(iphsta)
  deallocate(phamnt)
  deallocate(phasestr)



end subroutine tqex15_init

subroutine tqex15_calc(cmp,tl)
  use mtqex15
  implicit none
  double precision :: cmp(*),tl,ts
  integer :: i,ishort,ierr,noscheme


  ! move the compositions into the array together with the 
  ! condition for the pressure (temperature value is ignored, but
  ! a dummy value need to be given)
  do i=1,nel
     arr(i)=cmp(i)
  enddo
  arr(nel+1)=1500.0
  arr(nel+2)=101325.d0

  ! call the interpolation scheme to retrieve the temperature
  noscheme=0
  ierr=0
  ishort=0
  call tqips_get_value(ibranch,noscheme,arr, &
       tl,ierr,ishort,iwsg,iwse)

end subroutine tqex15_calc

subroutine tqex15_deinit
  use mtqex15
  implicit none
  ! deallocate some arrays used in the calculation
  deallocate(arr)
  deallocate(iwse)
  deallocate(iwsg)

end subroutine tqex15_deinit

program tqex15
  use mtqex15
  implicit none
  character :: liquid_phase*12
  character :: tcpath*256,tmppath*256
  integer :: i1,i2,i3,idiv,nstep,ierr,i
  double precision :: cmp(3),tl,eps
  logical sg1err
  integer :: icpu,jcpu
  eps=1.d-6

  ! define the name of the liquid phase
  liquid_phase='LIQ'

  ! allocate the workspaces for the TQ-library
  allocate(iwsg(TC_NWSG))
  allocate(iwse(TC_NWSE))

  ! initialize the TQ-library
  CALL GET_ENVIRONMENT_VARIABLE('TC24B_HOME',tcpath)
  ! use the default temporary directory
  tmppath=' '
  CALL TQINI3(tcpath,tmppath,TC_NWSG,TC_NWSE,IWSG,IWSE) 
  if (sg1err(ierr)) goto 999

  ! select a database
  call tqopdb('FEDEMO',iwsg,iwse)
  if (sg1err(ierr)) goto 999

  ! select the elements to be used in the calculation
  call tqdefel('C',iwsg,iwse)
  if (sg1err(ierr)) goto 999
  call tqdefel('CR',iwsg,iwse)
  if (sg1err(ierr)) goto 999
  call tqdefel('FE',iwsg,iwse)
  if (sg1err(ierr)) goto 999

  ! retrieve the data from the database
  call tqgdat(iwsg,iwse)
  if (sg1err(ierr)) goto 999

  ! initialize the interpolation and define the conditions for it
  call tqex15_init(liquid_phase)
  if (sg1err(ierr)) goto 999


  ! calculate an initial point
  cmp(1)=0.01D0
  cmp(2)=0.1D0
  cmp(3)=1.d0-cmp(1)-cmp(2)
  call tqex15_calc(cmp,tl)
  if (sg1err(ierr)) goto 999


  ! select a number for which results are to be displayed
  ! select a calculation spacing
  nstep=100
  idiv=nstep**2/10

  ! perform the calculation twice in order to compare execution
  ! time with and without initialization
  do i=1,2

     write(6,13)' pass ',i
     ! print a header
     write(6,10)'C','CR','FE','T-liq'

     call syscpu(1,icpu)
     i3=0
     ! loop in the C composition space
     do i1=0,nstep-1

        cmp(1)=0.1D0*dble(i1)/dble(nstep)
        if (cmp(1).le.0.d0) cmp(1)=eps
        if (cmp(1).ge.1.d0) cmp(1)=cmp(1)-eps

        ! loop in the CR composition space
        do i2=0,nstep-1

           cmp(2)=0.2D0*dble(i2)/dble(nstep)
           if (cmp(2).le.0.d0) cmp(2)=eps
           if (cmp(2).ge.1.d0) cmp(2)=cmp(2)-eps

           cmp(3)=1.0D0-cmp(1)-cmp(2)
           if (cmp(3).le.0.d0) cmp(3)=eps
           if (cmp(3).ge.1.d0) cmp(3)=cmp(3)-eps

           ! get the calculation result from the interpolation scheme
           call tqex15_calc(cmp,tl)
           if (sg1err(ierr)) goto 999

           ! display the result
           i3=i3+1
           if (mod(i3,idiv).eq.0) then
              write(6,12)i3,cmp(1),cmp(2),cmp(3),tl
           endif
        enddo
     enddo
     call syscpu(1,jcpu)
     write(0,13)'Cpu time ',jcpu-icpu
     call syscpu(1,icpu)
  enddo

  ! deinitialize the calculation
  call tqex15_deinit()
  if (sg1err(ierr)) goto 999

10 format(16X,3('X(',A,')',10X),A)
11 format(A,I8,A,$)
12 format(I13,1X,4(G13.6,1X))
13 format(1X,A,I3)
999 continue
end program tqex15
