    module module_tq_mpi_example

    integer, parameter :: nwsg=50000,nwse=50000,nwsw=50000

    integer, save :: iwsg(nwsg),iwse(nwse),iwsw(nwsw)

    end module module_tq_mpi_example
    !---------------------------------------------
    program TQ_MPI_Example

    ! This example show how MPI can be used in conjunction with TQ
    !
    ! NOTE: Thermo-Calc Software AB will not assist the user with MPI related 
    ! questions. This example is only meant to show how TQ can be used in
    ! applications together with MPI.
    !
    ! The example has been tested on Windows 10 64-bit using the Intel MPI runtime
    ! environment compliant with the MPI-3.1 standard and on linux (CentOS) 64-bit
    ! using Open MPI version 1.8 (version 1.7 or later is necessary).
    !
    ! To run the example it must first be compiled using an MPI library
    !
    ! The processes are divided up into a master (rank 0) and slaves (all other).
    ! The master process assigns and distributes tasks to the slave processes.
    !
    ! In this trivial example, a set of equilibrium calculations are distributed
    ! over all processes, the Gibbs energy of the system is retrieved and finally
    ! collected in a single vector in the master process.

    use module_tq_mpi_example

    implicit none
    include 'mpif.h'

    integer*4 runmpi_err,runmpi_rank,runmpi_size
    integer nel,neq_tot

    character*256 tcpath,gesfile

    gesfile='TQ_MPI_EXAMPLE'

    nel=3 ! Number of elements/components in the system.

    neq_tot=99  ! Compute in total 99 equilibria

    ! Initialize MPI
    call MPI_INIT(runmpi_err)
    ! Get the total number of MPI processes
    call MPI_COMM_SIZE(MPI_COMM_WORLD,runmpi_size,runmpi_err)
    ! Get the id (rank) of the calling process
    call MPI_COMM_RANK(MPI_COMM_WORLD,runmpi_rank,runmpi_err)
    write(*,*)'Process id ',runmpi_rank,' is alive'

    CALL GET_ENVIRONMENT_VARIABLE('TC61_HOME',tcpath)

    if (runmpi_rank.eq.0) then
        write(*,*)'Total number of processes: ',runmpi_size
        call TQ_MPI_Example_Master(runmpi_size,nel,neq_tot,tcpath,gesfile)
    else
        call TQ_MPI_Example_Slave(runmpi_rank,nel,tcpath,gesfile)
    endif


    end
    !------------------------------------
    subroutine TQ_MPI_Example_Master(runmpi_size,nel,neq_tot,tcpath,gesfile)

    use module_tq_mpi_example

    implicit none
    include 'mpif.h'


    integer*4 runmpi_size,runmpi_err,runmpi_flag,runmpi_count,runmpi_tag,runmpi_ok,runmpi_dest
    integer*4 runmpi_taskid,runmpi_count2,runmpi_root,runmpi_request,runmpi_neq,runmpi_rem
    integer*4 runmpi_request_array_length,runmpi_count3,runmpi_count4,runmpi_root2,runmpi_root3

    integer ierr,nel,neq_tot,length_variable_values,length_result_values,i,j,k,m,neq,err

    integer*4, allocatable :: runmpi_calculations_per_process(:),runmpi_number_sent_per_process(:),runmpi_displacements(:)
    integer*4, allocatable :: runmpi_number_received_per_process(:),runmpi_displacements_gather(:)
    integer*4, allocatable :: runmpi_request_array(:)
    integer*4 runmpi_stat(MPI_STATUS_SIZE)

    double precision, allocatable :: variable_values(:),variable_values_dummy_receive(:),variable_values_master(:)
    double precision, allocatable :: result_values(:),result_values_master(:)

    character*256 gesfile,tcpath,tmppath

    logical sg1err

    allocate(runmpi_calculations_per_process(runmpi_size))
    allocate(runmpi_number_sent_per_process(runmpi_size))
    allocate(runmpi_displacements(runmpi_size))
    allocate(runmpi_number_received_per_process(runmpi_size))
    allocate(runmpi_displacements_gather(runmpi_size))
    allocate(runmpi_request_array(3))
    runmpi_request_array_length=3

    ! initialize TQ for the master process
    ! use the default temporary directory
    tmppath=' '
    CALL TQINI3(tcpath,tmppath,nwsg,nwse,iwsg,iwse)
    if (sg1err(ierr)) then
        write(*,*)'Error initializing TQ'
        goto 999
    endif
    ! read data
    call tqrfil(gesfile,iwsg,iwse)
    if(sg1err(ierr)) then
        write(*,*)'error reading ges-file'
        goto 999
    endif
    write(*,*)'Master has initialized TQ'

    if (runmpi_size.gt.1) then
        ! Tell the slaves one at a time to initialize TQ
        runmpi_flag=1
        runmpi_count=1
        runmpi_tag=1
        do runmpi_dest=1,runmpi_size-1
            ! Send command to initialize TQ to slave process runmpi_dest
            call MPI_SEND(runmpi_flag,runmpi_count,MPI_INTEGER,runmpi_dest,runmpi_tag,MPI_COMM_WORLD,runmpi_err)
            ! Wait for signal that initialization is completed
            call MPI_RECV(runmpi_ok,runmpi_count,MPI_INTEGER,runmpi_dest,runmpi_tag,MPI_COMM_WORLD,runmpi_stat,runmpi_err)
            if (runmpi_ok.eq.0) then
                write(*,*)'Error initializing TQ'
                goto 999
            endif
        end do
    endif

    if (runmpi_size.gt.1) then
        ! Send work task #2 to slaves (compute equilibria)
        runmpi_taskid=2
        runmpi_count=1
        runmpi_root=0
        call MPI_BCAST(runmpi_taskid,runmpi_count,MPI_INTEGER,runmpi_root,MPI_COMM_WORLD,runmpi_err)
    endif

    if (runmpi_size.gt.1) then
        ! Send the number of equilibria that each slave should calculate
        runmpi_rem=mod(int(neq_tot,4),runmpi_size)
        runmpi_calculations_per_process=(int(neq_tot,4)-runmpi_rem)/runmpi_size
        if (runmpi_rem.gt.0) then
            do i=1,runmpi_rem
                runmpi_calculations_per_process(i)=runmpi_calculations_per_process(i)+1
            end do
        endif
        runmpi_count=1
        runmpi_count2=runmpi_count
        runmpi_root=0
        call MPI_ISCATTER(runmpi_calculations_per_process,runmpi_count,MPI_INTEGER,runmpi_neq,runmpi_count2, &
            MPI_INTEGER,runmpi_root,MPI_COMM_WORLD,runmpi_request_array(1),runmpi_err)
    else
        runmpi_calculations_per_process(1)=int(neq_tot,4)
    endif

    ! Send variable values
    ! Check allocation of arrays
    if (allocated(variable_values)) then
        if (length_variable_values.lt.(neq_tot*(nel+2))) then
            deallocate(variable_values)
            deallocate(variable_values_dummy_receive)
            deallocate(variable_values_master)
            length_variable_values=neq_tot*(nel+2)
            allocate(variable_values(length_variable_values))
            allocate(variable_values_dummy_receive(length_variable_values))
            allocate(variable_values_master(length_variable_values))
        endif
    else
        length_variable_values=neq_tot*(nel+2)
        allocate(variable_values(length_variable_values))
        allocate(variable_values_dummy_receive(length_variable_values))
        allocate(variable_values_master(length_variable_values))
    endif
    if (allocated(result_values)) then
        if (length_result_values.lt.(neq_tot+int(runmpi_size,8))) then
            deallocate(result_values)
            deallocate(result_values_master)
            length_result_values=neq_tot+int(runmpi_size,8)
            allocate(result_values(length_result_values))
            allocate(result_values_master(length_result_values))
        endif
    else
        length_result_values=neq_tot+int(runmpi_size,8)
        allocate(result_values(length_result_values))
        allocate(result_values_master(length_result_values))
    endif

    j=1
    do i=1,neq_tot
        ! The variable values are given in the sequence P,T,N_k
        variable_values(j)=1.0d5!Pressure
        j=j+1
        variable_values(j)=1.0d3!Temperature
        j=j+1
        variable_values(j)=1.0d0!Number of moles of Carbon
        j=j+1
        variable_values(j)=1.0d0!Number of moles of Chromium
        j=j+1
        variable_values(j)=0.01d0*dble(i)!Number of moles of Iron
        j=j+1
    end do
    do i=1,int(runmpi_calculations_per_process(1),8)*(nel+2)
        variable_values_master(i)=variable_values(i)
    end do
    do i=1,int(runmpi_size,8)
        runmpi_number_sent_per_process(i)=(int(nel,4)+2)*runmpi_calculations_per_process(i)
    end do
    if (runmpi_size.gt.1) then
        runmpi_displacements(1)=0
        do i=2,int(runmpi_size,8)
            runmpi_displacements(i)=runmpi_displacements(i-1)+runmpi_number_sent_per_process(i-1)
        end do
        runmpi_count3=runmpi_number_sent_per_process(1)
        runmpi_root2=0
        call MPI_ISCATTERV(variable_values,runmpi_number_sent_per_process,runmpi_displacements, &
        MPI_DOUBLE_PRECISION,variable_values_dummy_receive,runmpi_count3,MPI_DOUBLE_PRECISION,runmpi_root2, &
        MPI_COMM_WORLD,runmpi_request_array(2),runmpi_err)
    endif
    ! Compute the equilibria and acquire results
    result_values_master(1)=0.0d0
    neq=int(runmpi_calculations_per_process(1),8)
    call TQ_MPI_Example_compute_equilibria(neq,nel,variable_values_master,result_values_master(2),err)
    if (err.ne.0) then
        result_values_master(1)=1.0d0
    endif

    if (runmpi_size.gt.1) then
        ! Get results from slaves
        do i=1,int(runmpi_size,8)
            runmpi_number_received_per_process(i)=runmpi_calculations_per_process(i)+1
        end do
        runmpi_count4=runmpi_number_received_per_process(1)
        runmpi_displacements_gather(1)=0
        do i=2,int(runmpi_size,8)
            runmpi_displacements_gather(i)=runmpi_displacements_gather(i-1)+runmpi_number_received_per_process(i-1)
        end do
        runmpi_root3=0
        call MPI_IGATHERV(result_values_master,runmpi_count4,MPI_DOUBLE_PRECISION,result_values, &
            runmpi_number_received_per_process,runmpi_displacements_gather,MPI_DOUBLE_PRECISION, &
            runmpi_root3,MPI_COMM_WORLD,runmpi_request_array(3),runmpi_err)
        ! Wait for all non-blocking MPI calls to finish
        call MPI_WAITALL(runmpi_request_array_length,runmpi_request_array,MPI_STATUSES_IGNORE,runmpi_err)
    else
        do i=1,1+neq
            result_values(i)=result_values_master(i)
        end do
    endif

    ! Each process pass an error flag in the first position of their respective result_values vector.
    ! Check and remove these flags
    j=1
    k=0
    m=1
    do i=1,int(runmpi_size,8)+neq_tot
        if (k.eq.0) then
            if (result_values(i).eq.1.0d0) then
                write(*,*)'error'
            endif
        else
            result_values(j)=result_values(i)
            j=j+1
        endif
        k=k+1
        if (k.gt.int(runmpi_calculations_per_process(m),8)) then
            k=0
            m=m+1
        endif
    end do

    do i=1,neq_tot
        write(*,*) result_values(i)
    end do

    ! Send work task #1 to slaves (exit)
    if (runmpi_size.gt.1) then
        runmpi_taskid=1
        runmpi_count=1
        runmpi_root=0
        call MPI_BCAST(runmpi_taskid,runmpi_count,MPI_INTEGER,runmpi_root,MPI_COMM_WORLD,runmpi_err)
    endif


999 continue
    ! Finalize MPI
    call MPI_FINALIZE(runmpi_err)

    end
    !----------------------------------
    subroutine TQ_MPI_Example_Slave(runmpi_rank,nel,tcpath,gesfile)

    use module_tq_mpi_example

    implicit none
    include 'mpif.h'

    integer*4  i4,j4
    integer*4 runmpi_rank,runmpi_err,runmpi_count,runmpi_source,runmpi_flag,runmpi_tag,runmpi_ok,runmpi_dest
    integer*4 runmpi_taskid,runmpi_root,runmpi_neq,runmpi_request

    integer ierr,nel,length_variable_values,length_result_values,neq,err

    integer*4 runmpi_stat(MPI_STATUS_SIZE)

    double precision foo,foo2,foo3

    double precision, allocatable :: variable_values(:),result_values(:)

    character*256 tcpath,gesfile,tmppath

    logical sg1err

    ! Wait for a signal from the master before initializing TQ
    runmpi_count=1
    runmpi_source=0
    runmpi_tag=1
    call MPI_RECV(runmpi_flag,runmpi_count,MPI_INTEGER,runmpi_source,runmpi_tag,MPI_COMM_WORLD,runmpi_stat,runmpi_err)
    ! initialize TQ for the slave process
    ! use the default temporary directory
    runmpi_ok=1
    tmppath=' '
    CALL TQINI3(tcpath,tmppath,nwsg,nwse,iwsg,iwse)
    if (sg1err(ierr)) then
        call reserr
        write(*,*)'Error initializing TQ'
        runmpi_ok=0
    endif
    ! read data
    call tqrfil(gesfile,iwsg,iwse)
    if(sg1err(ierr)) then
        call reserr
        write(*,*)'error reading ges-file'
        runmpi_ok=0
    endif
    if (runmpi_ok.eq.1) then
        write(*,*)'slave ',runmpi_rank,' has initialized TQ'
    endif
    runmpi_dest=0
    runmpi_count=1
    runmpi_tag=1
    ! Signal to master that initialization is complete (or has failed)
    call MPI_SEND(runmpi_ok,runmpi_count,MPI_INTEGER,runmpi_dest,runmpi_tag,MPI_COMM_WORLD,runmpi_err)
    if (runmpi_ok.eq.0) then
        goto 999
    endif

10  continue
    ! Wait for work task
    runmpi_count=1
    runmpi_root=0
    call MPI_BCAST(runmpi_taskid,runmpi_count,MPI_INTEGER,runmpi_root,MPI_COMM_WORLD,runmpi_err)
    goto(100,200),runmpi_taskid
    ! tasks:
    ! 1. Exit
    ! 2. Compute equilibria

100 continue
    write(*,*)'Slave ',runmpi_rank,' received signal to exit'
    goto 999

200 continue
    ! Get the number of equilibria to be calculated
    runmpi_count=1
    runmpi_root=0
    call MPI_ISCATTER(i4,j4,MPI_INTEGER,runmpi_neq,runmpi_count,MPI_INTEGER,runmpi_root,MPI_COMM_WORLD,runmpi_request,runmpi_err)

    call MPI_WAIT(runmpi_request,MPI_STATUSES_IGNORE,runmpi_err)

    write(*,*)'slave #',runmpi_rank,' will calculate ',runmpi_neq,' equilibria'

    ! Check allocation of arrays
    neq=int(runmpi_neq,8)
    if (allocated(variable_values)) then
        if (length_variable_values.lt.(neq*(nel+2))) then
            deallocate(variable_values)
            length_variable_values=neq*(nel+2)
            allocate(variable_values(length_variable_values))
        endif
    else
        length_variable_values=neq*(nel+2)
        allocate(variable_values(length_variable_values))
    endif
    if (allocated(result_values)) then
        if (length_result_values.lt.(neq+1)) then
            deallocate(result_values)
            length_result_values=neq+1
            allocate(result_values(length_result_values))
        endif
    else
        length_result_values=neq+1
        allocate(result_values(length_result_values))
    endif

    ! Get the variable values
    runmpi_count=(int(nel,4)+2)*runmpi_neq
    runmpi_root=0
    call MPI_ISCATTERV(foo,foo2,foo3, &
        MPI_DOUBLE_PRECISION,variable_values,runmpi_count,MPI_DOUBLE_PRECISION,runmpi_root,MPI_COMM_WORLD,runmpi_request,runmpi_err)

    call MPI_WAIT(runmpi_request,MPI_STATUSES_IGNORE,runmpi_err)

    write(*,*)'slave #',runmpi_rank,' received variable values'

    ! Compute the equilibria and acquire results
    result_values(1)=0.0d0
    call TQ_MPI_Example_compute_equilibria(neq,nel,variable_values,result_values(2),err)
    if (err.ne.0) then
        result_values(1)=1.0d0
    endif

    ! Return results (only one value is returned for each calculated equilibria)
    runmpi_count=runmpi_neq+1
    runmpi_root=0
    call MPI_IGATHERV(result_values,runmpi_count,MPI_DOUBLE_PRECISION,foo, &
            i4,j4,MPI_DOUBLE_PRECISION,runmpi_root,MPI_COMM_WORLD,runmpi_request,runmpi_err)

    call MPI_WAIT(runmpi_request,MPI_STATUSES_IGNORE,runmpi_err)

    goto 10

999 continue
    ! Finalize MPI
    call MPI_FINALIZE(runmpi_err)

    end
    !----------------------------
    subroutine TQ_MPI_Example_compute_equilibria(neq,nel,variable_values,result_values,err)


    use module_tq_mpi_example

    implicit none

    integer neq,nel,err,i,j,k,m,ierr

    double precision variable_values(*),result_values(*)

    logical sg1err

    err=0
    j=1
    do i=1,neq
        call tqsetc('P',-1,-1,variable_values(j),k,iwsg,iwse)
        j=j+1
        call tqsetc('T',-1,-1,variable_values(j),k,iwsg,iwse)
        j=j+1
        do m=1,nel
            call tqsetc('N',-1,m,variable_values(j),k,iwsg,iwse)
            j=j+1
        end do
        call tqceg(iwsg,iwse)
        call tqget1('G',-1,-1,result_values(i),iwsg,iwse)
    end do


    end

