/* TQ library example to illustrate the use the adaptive interpolation */
/* scheme. */
/* This example calculates the liquidus temperature in a part of the */
/* C-CR-FE system and displays a selection of the results. */

#define _CRT_SECURE_NO_WARNINGS

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>
#include "tqroot.h"
#include "tcutils.h"

TC_INT *iwsg,*iwse;
TC_FLOAT *arr;
TC_INT nel,ibranch;

void tqex15_init(TC_STRING liquid_name)
{
  TC_INT i,nph,ill,ierr,imb,iph_liquid;
  TC_INT *idepel,*iphsta;
  TC_STRING phase_name,str;
  tc_phases_strings *phasestr;
  TC_BOOL tiscnd,tiscst,piscnd,piscst;
  TC_INT idisct,logstep;
  TC_FLOAT t,tmin,tmax,p,pmin,pmax,tmemfrac;
  TC_FLOAT *xmin,*xmax,*phamnt;


  /* get the number of components */
  tq_gnc(&nel,iwsg,iwse);
  if (tq_sg1err(&ierr)){
    fprintf(stderr,"error:%d\n",ierr);
    return;
  }

  /* get the number of phases */
  tq_gnp(&nph,iwsg,iwse);
  if (tq_sg1err(&ierr)){
    fprintf(stderr,"error:%d\n",ierr);
    return;
  }


  /* allocate arrays needed for the calculation */
  str=malloc(TC_STRLEN_MAX);
  phase_name=malloc(TC_STRLEN_PHASES);

  idepel=malloc(sizeof(TC_BOOL)*nel);
  xmin=malloc(sizeof(TC_FLOAT)*nel);
  xmax=malloc(sizeof(TC_FLOAT)*nel);
  arr=malloc(sizeof(TC_FLOAT)*(nel+2));
  phasestr=malloc(sizeof(tc_phases_strings)*nph);
  iphsta=malloc(sizeof(TC_INT)*nph);
  phamnt=malloc(sizeof(TC_FLOAT)*nph);

  /* find the index of the liquid phase and append '#1' to all phases */
  ill=strlen(liquid_name);
  for (i=0; i<nph; i++) {
    tq_gpn(i+1,phasestr[i].phase,TC_STRLEN_PHASES,iwsg,iwse);
    if (tq_sg1err(&ierr)){
      fprintf(stderr,"error:%d\n",ierr);
      return;
    }
    if (strstr(phasestr[i].phase,"#")==NULL) {
      strcat(phasestr[i].phase,"#1");
    }
    if (strncmp(phasestr[i].phase,liquid_name,ill) == 0 ){
      iph_liquid=i;
    }
  }

  /* initialize the interpolation scheme */
  ierr=0;
  tq_ips_init_top(&ierr,iwsg,iwse);

  /* define that temperature varies in this calculation */
  /* (is neither constant nor a condition) */
  tiscnd=false;
  tiscst=false;
  /* define that pressure is fixed in this calculation */
  /* (is a condition and do not vary) */
  piscnd=true;
  piscst=true;

  /* define that all compositions are given */
  /* (the content of all components are conditions) */
  for (i=0; i< nel; i++){
    idepel[i]=true;
  }

  /* use a linear discretization in composition space */
  idisct=1;
  /* use a virtual spacing corresponding to 100 points in composition  */
  /* and temperature space */
  logstep=2;
  /* assume a minimum, maximum and start value for temperature */
  tmin=1800.0;
  tmax=2000.0;
  t=(tmin+tmax)*0.5;
  /* assume a minimum, maximum and start value for pressure */
  p=101325.0;
  pmin=101325.0;
  pmax=101325.0;
  /* allow the interpolation scheme to use a maximum of 70% of the  */
  /* free memory */
  tmemfrac=0.7;


  /* retrieve the amount of free memory */
  tq_gfmm(&imb);
  /* set the maximum memory allocation to be 1400Mb on a 32bit machine */
  if (sizeof(char*) == 4 ) {
    if (tmemfrac > 100.0/(TC_FLOAT)imb ) {
      tmemfrac=100.0/(TC_FLOAT)imb;
    }
  }

  /* set the minimum and maximum range of the composition of the */
  /* components */
  for (i=0; i< nel; i++){
    xmin[i]=0.0;
    xmax[i]=1.0;
  }


  /* set all phases ENTERED except the liquid phase, which is to */
  /* be FIXED with amount 1 */
  for (i=0; i<nph; i++) {
    iphsta[i]=1;
    phamnt[i]=0.0;
  }
  iphsta[iph_liquid]=4;
  phamnt[iph_liquid]=1.0;

  /* initialize this branch in the interpolation scheme */
  ierr=0;
  tq_ips_init_branch(tiscnd,tiscst,piscnd,piscst,
		 idepel,
		 idisct,logstep,
		 iphsta,
		 t,tmin,tmax,p,pmin,pmax,tmemfrac,
		 phamnt,xmin,xmax,
		 &ibranch,&ierr,
		 iwsg,iwse);
  
  /* initialize a function to retrieve the temperature */
  strcpy(str,"T");  
  ierr=0;
  tq_ips_init_function(str,ibranch,&ierr,iwsg,iwse);
  
  
  /* deallocate temporary variables needed for the setup */
  free(idepel);
  free(xmin);
  free(xmax);
  free(iphsta);
  free(phamnt);
  free(phasestr);
}


void tqex15_calc(TC_FLOAT* cmp,
		 TC_FLOAT* tl)
{
  TC_INT i,ishort,ierr,noscheme;

  /* move the compositions into the array together with the  */
  /* condition for the pressure (temperature value is ignored, but */
  /* a dummy value need to be given) */
  for (i=0; i< nel; i++){
    arr[i]=cmp[i];
  }
  arr[nel]=1500;
  arr[nel+1]=101325.0;

  /* call the interpolation scheme to retrieve the temperature */
  noscheme=0;
  ierr=0;
  ishort=0;
  tq_ips_get_value(ibranch,noscheme,arr,
	       tl,&ierr,&ishort,iwsg,iwse);
}

void tqex15_deinit()
{
  /* deallocate some arrays used in the calculation */
  free(arr);
  free(iwse);
  free(iwsg);
}



main(int argc, char *argv)
{
  TC_STRING liquid_phase;
  TC_INT i1,i2,i3,idiv,nstep,ierr,i;
  TC_INT icpu,jcpu;
  TC_FLOAT *cmp,*tl,eps;
  div_t div_result;
  char log_file_directory[FILENAME_MAX];
  char tc_installation_directory[FILENAME_MAX];
  eps=1e-6;

  liquid_phase=malloc(12);
  cmp=malloc(sizeof(TC_FLOAT)*3);
  tl=malloc(sizeof(TC_FLOAT));

  /* define the name of the liquid phase */
  strcpy(liquid_phase,"LIQ");

  /* allocate the workspaces for the TQ-library */
  iwsg=malloc(sizeof(TC_INT)*TC_NWSG);
  iwse=malloc(sizeof(TC_INT)*TC_NWSE);

  memset(iwsg,0,sizeof(TC_INT)*TC_NWSG);
  memset(iwse,0,sizeof(TC_INT)*TC_NWSE);
  memset(log_file_directory,0,sizeof(log_file_directory));
  memset(tc_installation_directory,0,sizeof(tc_installation_directory));

  /* initialize the TQ-library */
  
  /*Find the necessary path for log file output. (Defaulted to TEMP directory)*/
  getTempEnvironmentPath(log_file_directory);
  /*Find the path to databases. (Defaulted to Thermo-Calc installation directory)*/
  getThermoCalcEnvironmentPath(tc_installation_directory);

  tq_ini3(tc_installation_directory,log_file_directory,TC_NWSG,TC_NWSE,iwsg,iwse);
  if (tq_sg1err(&ierr)){
    fprintf(stderr,"error:%d\n",ierr);
    return;
  }

  /* select a database */
  tq_opdb("FEDEMO",iwsg,iwse);
  if (tq_sg1err(&ierr)){
    fprintf(stderr,"error:%d\n",ierr);
    return;
  }

  /* select the elements to be used in the calculation */
  tq_defel("C",iwsg,iwse);
  if (tq_sg1err(&ierr)){
    fprintf(stderr,"error:%d\n",ierr);
    return;
  }
  tq_defel("CR",iwsg,iwse);
  if (tq_sg1err(&ierr)){
    fprintf(stderr,"error:%d\n",ierr);
    return;
  }
  tq_defel("FE",iwsg,iwse);
  if (tq_sg1err(&ierr)){
    fprintf(stderr,"error:%d\n",ierr);
    return;
  }

  /* retrieve the data from the database */
  tq_gdat(iwsg,iwse);
  if (tq_sg1err(&ierr)){
    fprintf(stderr,"error:%d\n",ierr);
    return;
  }

  /* initialize the interpolation and define the conditions for it */
  tqex15_init(liquid_phase);
  if (tq_sg1err(&ierr)){
    fprintf(stderr,"error:%d\n",ierr);
    return;
  }

  /* calculate an initial point */
  cmp[0]=0.01;
  cmp[1]=0.1;
  cmp[2]=1.0-cmp[0]-cmp[1];
  tqex15_calc(cmp,tl);
  if (tq_sg1err(&ierr)){
    fprintf(stderr,"error:%d\n",ierr);
    return;
  }

  /* select a number for which results are to be displayed */
  /* select a calculation spacing */
  idiv=1000;
  nstep=100;

  /* perform the calculation twice in order to compare execution */
  /* time with and without initialization */
  for (i=0; i<2; i++){

    fprintf(stdout,"pass %d\n",i+1);

    /* print a header */
    fprintf(stdout,"        X(C)     X(CR)    X(FE)   T-liq\n");

    tq_syscpu(1,&icpu);
    i3=0;
    /* loop in the C composition space */
    for (i1=0; i1 < nstep; i1++) {
      cmp[0]=0.1*(TC_FLOAT)i1/(TC_FLOAT)nstep;
      if (cmp[0] <= 0.0 ) { cmp[0]=eps;}
      if (cmp[0] >= 1.0 ) { cmp[0]=cmp[0]-eps;}

      /* loop in the CR composition space */
      for (i2=0; i2 < nstep; i2++) {
	cmp[1]=0.2*(TC_FLOAT)i2/(TC_FLOAT)nstep;
	if (cmp[1] <= 0.0 ) { cmp[1]=eps;}
	if (cmp[1] >= 1.0 ) { cmp[1]=cmp[1]-eps;}

	cmp[2]=1.0-cmp[0]-cmp[1];
	if (cmp[2] <= 0.0 ) { cmp[2]=eps;}
	if (cmp[2] >= 1.0 ) { cmp[2]=cmp[2]-eps;}

	/* get the calculation result from the interpolation scheme */
	tqex15_calc(cmp,tl);
	if (tq_sg1err(&ierr)){
	  fprintf(stderr,"error:%d\n",ierr);
	  return;
	}
	/* display the result */
	i3++;
	div_result=div((int)i3,(int)idiv);
	if ( div_result.rem == 0 ) {	
	  fprintf(stdout,"%d %f %f %f %f\n",i3,cmp[0],cmp[1],cmp[2],*tl);
	}
      }
    }
    tq_syscpu(1,&jcpu);
    fprintf(stderr,"Cpu time %d\n",jcpu-icpu);
    tq_syscpu(1,&icpu);
  }
  /* deinitialize the calculation */
  tqex15_deinit();
  if (tq_sg1err(&ierr)){
    fprintf(stderr,"error:%d\n",ierr);
    return;
  }
}
