/************************************************************************/
/*                                                                      */
/*    This sample program shows how to retrieve data from a Thermo-     */
/*    Calc data file, then define a set of conditions for a single      */
/*    equilibrium calculation, get the equilibrium phases and their     */
/*    amounts and compositions. The method of calculating the liquidus  */
/*    and solidus temperature is also demonstrated.                     */
/*                                                                      */
/************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include "tqroot.h"


void writepx(TC_INT iph,TC_INT* iwsg,TC_INT* iwse)
{
  TC_STRING phname,sname;
  TC_INT i;
  TC_FLOAT val,valnp,valw;

  phname=malloc(TC_STRLEN_PHASES);
  sname=malloc(TC_STRLEN_MAX);

  fprintf(stdout,"phase name    composition, Wt percent Cu     amount\n");
  for (i=0; i<iph; i++) {
    tq_get1("DG", i+1, -1, &val, iwsg,iwse);
    if(val == 0.0) {
      tq_gpn(i+1, phname, TC_STRLEN_PHASES, iwsg,iwse);
      tq_get1("np", i+1, -1, &valnp, iwsg,iwse);
      tq_get1("w%", i+1,  2, &valw,  iwsg,iwse);
      if (valnp < TC_EPS) { valnp=0.0;}
      fprintf(stdout,"%8s          %16g %16g\n",phname,valw,valnp);
    }
  }
  fprintf(stdout,"\n");

  free(phname);
  free(sname);
}





main(int argc, char *argv)
{

  TC_INT *iwsg,*iwse;
  tc_components_strings *components;
  TC_INT i,ncomp,iph,iliq;
  TC_INT icont,iconn,iconp,iconw;
  TC_BOOL pstat;
  TC_STRING phname,sname;
  TC_FLOAT an,val,valt;
  TC_INT ipp,ipm,imc;
  TC_FLOAT vm,vp,surface_energy;
  TC_FLOAT* x;
  TC_INT ierr;
  char log_file_directory[FILENAME_MAX];
  char tc_installation_directory[FILENAME_MAX];
  
  ierr = 0;
  iwsg=malloc(sizeof(TC_INT)*TC_NWSG);
  iwse=malloc(sizeof(TC_INT)*TC_NWSE);
  phname=malloc(TC_STRLEN_PHASES);
  sname=malloc(TC_STRLEN_MAX);
  components=malloc(TC_MAX_NR_OF_ELEMENTS*sizeof(tc_components_strings));

  memset(iwsg,0,sizeof(TC_INT)*TC_NWSG);
  memset(iwse,0,sizeof(TC_INT)*TC_NWSE);
  memset(log_file_directory,0,sizeof(log_file_directory));
  memset(tc_installation_directory,0,sizeof(tc_installation_directory));
    
  /* initiate the workspace */
  tq_ini3(tc_installation_directory,log_file_directory,TC_NWSG,TC_NWSE,iwsg,iwse);

  /* read the thermodynamic data file which was created by using */
  /* the GES module inside the Thermo-Calc software package */
  tq_rfil("TQEX01",iwsg,iwse);
  if(tq_sg1err(&ierr))return;

  /* get component names in the system */
  tq_gcom(&ncomp,components,iwsg,iwse);
  if(tq_sg1err(&ierr))return;

  x=malloc(ncomp*sizeof(TC_FLOAT));

  fprintf(stdout,"This system has the following components:\n");
  for (i=0; i < ncomp; i++) {
    fprintf(stdout,"%d %s\n",i+1,components[i].component);
  }
  fprintf(stdout,"\n");

  /* get number of phases in the system */
  tq_gnp(&iph, iwsg,iwse);
  if(tq_sg1err(&ierr))return;
  fprintf(stdout,"This system has %d phases:\n",iph);
  if(tq_sg1err(&ierr))return;
  /* get names and status of the phases in the system */
  for (i=0; i<iph; i++) {
    tq_gpn(i+1, phname, TC_STRLEN_PHASES, iwsg,iwse);
    if(tq_sg1err(&ierr))return;
    pstat=tq_gsp(i+1, sname, TC_STRLEN_MAX, &an, iwsg,iwse);
    fprintf(stdout,"%s %s %g\n",phname,sname, an);
  }
  fprintf(stdout,"\n");

  /* set the units of some properties. */
  tq_ssu((TC_STRING)"ENERGY",(TC_STRING)"CAL",iwsg,iwse);
  if(tq_sg1err(&ierr))return;
  tq_ssu((TC_STRING)"T",(TC_STRING)"C",iwsg,iwse);
  if(tq_sg1err(&ierr))return;

  /* set the condition for a sigle equilibrium calculation */
  tq_setc("T",-1,-1,630.0,&icont,iwsg,iwse);
  tq_setc("N", -1, -1, 1.00, &iconn, iwsg,iwse);
  tq_setc("P",-1,-1,101325.0,&iconp,iwsg,iwse);
  tq_setc("W%",-1, 2, 8.0, &iconw, iwsg,iwse);
  if(tq_sg1err(&ierr))return;

  /* calculate equilibrium */
  tq_ce(" ", 0, 0, 0.0, iwsg,iwse);
  if(tq_sg1err(&ierr))return;
  tq_get1("GM",-1,-1,&val,iwsg,iwse);
  fprintf(stdout,"At T = 630.0 C, Wt percent Cu = 8\n");
  fprintf(stdout,"The Gibbs energy of the system is %g Cal/mol.\n",val);
  /* find the equilibrium phase(s) and their compositions */
  writepx(iph,iwsg,iwse);

  /* find the equilibrium liquidus and solidus temperature of this alloy */
  /*  */
  /* change the status of the liquid phase to FIXED with amount 1 */
  /* and then remove the temperature condition */
  tq_gpi(&iliq,"liq",iwsg,iwse);
  tq_csp(iliq, "FIXED", 1.0, iwsg,iwse);
  tq_remc(icont,iwsg,iwse);
  if(tq_sg1err(&ierr))return;
  tq_ce(" ",0,0,0.0,iwsg,iwse);
  if(tq_sg1err(&ierr))return;
  tq_get1("T",-1,-1,&valt,iwsg,iwse);
  fprintf(stdout,"Tliquidus = %gC\n", valt);
  /* find the equilibrium phase(s) and composition(s) */
  writepx(iph,iwsg,iwse);

  /* change the status of the liquid phase to FIXED with amount 0 */
  tq_csp(iliq, "FIXED", 0.0, iwsg,iwse);
  if(tq_sg1err(&ierr))return;
  tq_ce(" ",0,0,0.0,iwsg,iwse);
  if(tq_sg1err(&ierr))return;
  tq_get1("T",-1,-1,&valt,iwsg,iwse);
  fprintf(stdout,"Tsolidus = %gC\n", valt);
  /* find the equilibrium phase(s) and composition(s) */
  writepx(iph,iwsg,iwse);

  /* find the temperature where the equilibrium solidification is half done */
  tq_csp(iliq, "FIXED", 0.5, iwsg,iwse);
  if(tq_sg1err(&ierr))return;
  tq_ce(" ",0,0,0.0,iwsg,iwse);
  if(tq_sg1err(&ierr))return;
  tq_get1("T",-1,-1,&valt,iwsg,iwse);
  fprintf(stdout,"Tmid = %gC\n", valt);
  /* find the equilibrium phase(s) and composition(s) */
  writepx(iph,iwsg,iwse);


  tq_gpi(&ipp,"FCC_A1",iwsg,iwse);
  tq_gpi(&ipm,"LIQUID",iwsg,iwse);
  tq_gsci(&imc,"AL",iwsg,iwse);

  x[0]=0.92;
  x[1]=0.08;
  valt=873.16;
  vm=0.9e-5;
  vp=0.9e-5;

  surface_energy = tq_gse(ipm,ipp,imc,valt,x,vm,vp,iwsg,iwse);
  if(tq_sg1err(&ierr))return;

  free(components);
  free(phname);
  free(sname);
  free(x);
  free(iwsg);
  free(iwse);
}
