/************************************************************************/
/*                                                                      */
/*    This sample program demonstrates that the number of phases can    */
/*    increase due to the use of global mininization for equilibrium    */
/*    calculation during which additional composition set(s) can be     */
/*    added automatically if a miscibility gap is detected.             */
/*                                                                      */
/************************************************************************/

#define _CRT_SECURE_NO_WARNINGS

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>
#include "tqroot.h"
#include "tcutils.h"


void prphase(TC_INT* iwsg, TC_INT* iwse)
{
  TC_STRING phas1;
  TC_INT npha,i;

  phas1=malloc(TC_STRLEN_PHASES);

  tq_gnp(&npha,iwsg,iwse);
  fprintf(stdout,"number of phases : %d\n",npha);
  for (i=0; i<npha; i++) {
    tq_gpn(i+1,phas1,TC_STRLEN_PHASES,iwsg,iwse);
    fprintf(stdout,"index of %s is %d\n",phas1,i+1);
  }
  free(phas1);
}

	
main(int argc, char *argv)
{

  TC_INT *iwsg,*iwse;
  TC_INT ierr;

  TC_INT icmp2,numcon;
  TC_FLOAT p,n,t,xfcr;

  char log_file_directory[FILENAME_MAX];
  char tc_installation_directory[FILENAME_MAX];

  iwsg=malloc(sizeof(TC_INT)*TC_NWSG);
  iwse=malloc(sizeof(TC_INT)*TC_NWSE);

  memset(iwsg,0,sizeof(TC_INT)*TC_NWSG);
  memset(iwse,0,sizeof(TC_INT)*TC_NWSE);
  memset(log_file_directory,0,sizeof(log_file_directory));
  memset(tc_installation_directory,0,sizeof(tc_installation_directory));

  /*  initiate workspace */

  /*Find the necessary path for log file output. (Defaulted to TEMP directory)*/
  getTempEnvironmentPath(log_file_directory);
  /*Find the path to databases. (Defaulted to Thermo-Calc installation directory)*/
  getThermoCalcEnvironmentPath(tc_installation_directory);

  tq_ini3(tc_installation_directory,log_file_directory,TC_NWSG,TC_NWSE,iwsg,iwse);

  /*  open database FEDEMO */
  tq_opdb("FEDEMO",iwsg,iwse);
  if (tq_sg2err(&ierr)){
    fprintf(stdout,"Database or database license not available!\n");
    tq_reserr();
  }
  
  /*  define element Fe */
  tq_defel("Fe",iwsg,iwse);
  if (tq_sg2err(&ierr)){
    fprintf(stdout,"Error in defining element Fe!\n");
    tq_reserr();
  }
  /*  define element Cr */
  tq_defel("Cr",iwsg,iwse);
  if (tq_sg2err(&ierr)){
    fprintf(stdout,"Error in defining element Cr!\n");
    tq_reserr();
  }

  /*  get data from the database */
  tq_gdat(iwsg,iwse);
  if (tq_sg2err(&ierr)){
    fprintf(stdout,"Error in getting data from database\n");
    return;
  }

  /*  print phase names and their index */
  /*  note no 2nd composition set added manually for BCC phase */
  prphase(iwsg,iwse);



  /*  set conditions */
  p=101325.0;
  n=1.0;
  xfcr=0.5;
  tq_setc("P",-1,-1,p,&numcon,iwsg,iwse);
  if (tq_sg2err(&ierr)){
    fprintf(stdout,"error setting condition\n");
    return;
  }
  tq_setc("N",-1,-1,n,&numcon,iwsg,iwse);
  if (tq_sg2err(&ierr)){
    fprintf(stdout,"error setting condition\n");
    return;
  }
  tq_gsci(&icmp2,"CR",iwsg,iwse);
  if (tq_sg2err(&ierr)){
    fprintf(stdout,"error getting index\n");
    return;
  }
  tq_setc("X",-1,icmp2,xfcr,&numcon,iwsg,iwse);
  if (tq_sg2err(&ierr)){
    fprintf(stdout,"error setting condition\n");
    return;
  }

  /*  make calculations at 1000 K, where there should */
  /*  be no miscibility gap exsiting. As a result, normal */
  /*  calculation and global one should give same result. */
  t=1000;
  tq_setc("T",-1,-1,t,&numcon,iwsg,iwse);
  if (tq_sg2err(&ierr)){
    fprintf(stdout,"error setting condition\n");
    return;
  }
  tq_ce(" ",0,0,0.0,iwsg,iwse);
  if (tq_sg2err(&ierr)){
    fprintf(stdout,"error calculating\n");
    return;
  }

  /*  open the screen output unit and list equilibrium result */
  /*  correct equilibbrium obtained. */
  tq_sio("output",6);
  fflush(stdout);
  tq_le(iwsg,iwse);
  fflush(stdout);

  /*  print phase names and their index */
  /*  nothing changed */
  prphase(iwsg,iwse);
  fflush(stdout);
  
  /*  try global minimization */
  tq_ceg(iwsg,iwse);
  /*  list equilibrium result, which is the same as before */
  tq_le(iwsg,iwse);
  fflush(stdout);


  /*  print phase names and their index. nothing changed */
  prphase(iwsg,iwse); 
  fflush(stdout);

  /*  make calculations at 500 K, where BCC should split into */
  /*  two BCC phases, one rich in Fe, and the other in Cr. */
  /*  normal calculation will end up with a metastable equilibrium */
  /*  because no 2nd composition set predefined. */
  /*  global minimization algorithm can automatically generate */
  /*  this 2nd composition set if found necessary and hence */
  /*  garantee that the stable equilibrium will be obtained. */
  t=500.0;
  tq_setc("T",-1,-1,t,&numcon,iwsg,iwse);
  tq_ce(" ",0,0,0.0,iwsg,iwse);
  /*  list result. a metastable equilibrium instead stable equilibrium */
  /*  is obtained */
  tq_le(iwsg,iwse);

  /*  print phase names and their index. nothing changed */
  prphase(iwsg,iwse);
  fflush(stdout);

  /*  try global and list calculation result. the stable  */
  /*  equilibrium is obtained. */
  tq_ceg(iwsg,iwse);
  tq_le(iwsg,iwse);
  fflush(stdout);

  /*  print phase names and their index. */
  /*  number of system phases increased by one due to the */
  /*  appearance of BCC#2. note his "new" phase is added */
  /*  to the bottom of the list and the index of "old" phases */
  /*  is unaltered. */
  prphase(iwsg,iwse);
  fflush(stdout);


  free(iwse);
  free(iwsg);
}
