%{
Calculates and plots the single phase FCC_A1 region of Al-Si-Cu alloys and compares them for different Cu-contents.
For simplicity only FCC_A1 and DIAMOND_A4 are enabled as solid phases.
%}
database = "ALDEMO";
dependent_element = "Al";
compositions = containers.Map({'Si', 'Cu'}, ...% in wt-%
                              {9.0 , 3.0});
x_axis_element = "Si";
x_axis_min = 0;       % wt-%
x_axis_max = 2.5;     % wt-%
temp_axis_min = 300;  % in K
temp_axis_max = 1000; % in K
varied_element = "Cu";
variations = [1.0, 2.0, 3.0];  % in wt-%
plot_colors = ["red",  "green", "blue"];
temp_ref = 1000;  % temperature for the initial equilibrium, in K

% needs to be determined in advanced by checking the labels in the result data
relevant_labels = ["DIAMOND_A4", "LIQUID"];

session = tc_toolbox.TCToolbox();

[filepath,name,ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder( name + "_cache");

elements_solutes = string(compositions.keys());
elements = [dependent_element elements_solutes];
tc_system = session.select_database_and_elements(database, elements)...
    .without_default_phases()...
    .select_phase("FCC_A1")...
    .select_phase("DIAMOND_A4")...
    .select_phase("LIQUID")...
    .get_system();
                

calculator = tc_system.with_phase_diagram_calculation();
calculator.set_condition(tc_toolbox.ThermodynamicQuantity.temperature(), ...
    temp_ref);

for element = elements_solutes
    calculator.set_condition(...
        tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(element),...
        compositions(element) / 100);
end

for variation_index = 1:length(variations)
    varied_element_content = variations(variation_index);
    calculator.set_condition(...
        tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(varied_element),...
        varied_element_content / 100);

    phase_diagram = calculator...
        .with_first_axis(tc_toolbox.step_or_map_diagrams.CalculationAxis(...
            tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(x_axis_element))...
                .set_min(x_axis_min / 100)...
                .set_max(x_axis_max / 100)...
        ).with_second_axis(tc_toolbox.step_or_map_diagrams.CalculationAxis(...
            tc_toolbox.ThermodynamicQuantity.temperature())...
                .set_min(temp_axis_min)...
                .set_max(temp_axis_max)...
        ).calculate();

    % add labels for better readability
    % in the single FCC_A1 region for all variations
    phase_diagram.add_coordinate_for_phase_label(0.3, 790.0);  
    
    % in the DIAMOND_A4 + FCC_A1 region for all variations
    phase_diagram.add_coordinate_for_phase_label(1.2, 755);
    
    % in the FCC_A1 + LIQUID region for all variations
    phase_diagram.add_coordinate_for_phase_label(0.5, 900.0);  
 
    result_data = ...
        phase_diagram.get_values_grouped_by_stable_phases_of(...
            "M-P " + x_axis_element,...
            tc_toolbox.ThermodynamicQuantity.temperature()...
        );
              
    lines = result_data.get_lines();

    for line_index = 1:length(relevant_labels)
        label = relevant_labels(line_index);
        line = lines(label);
        if line_index == 1 % Only show legend item for the first line 
           plot_handles(variation_index) = plot(line.get_x, line.get_y, ...
               "Color", plot_colors(variation_index));
        else
           plot(line.get_x(), line.get_y(), "Color", ...
               plot_colors(variation_index)); 
        end         
        hold on
    end

    phase_region_labels = result_data.get_phase_labels();
    for label_index = 1 : length(phase_region_labels)
        label = phase_region_labels{label_index};
        curr_text_object = text(label.get_x(), label.get_y(), ...
            label.get_text());
        set(curr_text_object, "Interpreter", "none");
    end
end
legend(plot_handles, varied_element + "=" + variations + " wt-%");
hold off

legend;
title("Al-Si9-Cu3 system");
xlabel(x_axis_element + " [wt-%]");
ylabel("T [K]");
xlim([x_axis_min x_axis_max]);
ylim([temp_axis_min temp_axis_max]);
