%{
This example shows how to perform step calculations for a multi-material 
system. Such a calculation is not possible using the Thermo-Calc Graphical 
User Interface.
%}
dependent_element = "Fe";
elements_input = ["Cr", "Ni", "C", "Mn"];

alloy_A_input = containers.Map({'Cr', 'Ni',  'C',    'Mn'}, ...
                               { 0.1 , 0.02, 0.023,  0.11 });
alloy_B_input = containers.Map({'Cr', 'Ni',  'C',    'Mn'}, ...
                               { 0.3 , 0.43,  0.01,  0.22 });                               

conditions = containers.Map({'T'}, {1000.0});

% Create new variables for elements, alloy A and B that skip elements that 
% do no change in composition between alloy A and B. Add the remaining 
% elements as conditions directly.
alloy_A = containers.Map();
alloy_B = containers.Map();
elements = [];
for element = elements_input
    if alloy_A_input(element) == alloy_B_input(element)
        condition = "W(" + element + ")";
        conditions(condition) = alloy_A_input(element);
    else
        alloy_A(element) = alloy_A_input(element);
        alloy_B(element) = alloy_B_input(element);
        elements = [elements element];
    end
end

% Always take the last element to be the one that we vary in step calculation
axis_element = elements(end);

% the step/map variable always need to increase, swap the alloys if that 
% is not the case.
if alloy_A(axis_element) > alloy_B(axis_element)
    temp = alloy_A;
    alloy_A = alloy_B;
    alloy_B = temp;
end

% Create the conditions for all dependent elements so they all linearly 
% depend on the axis_element condition.
% The conditions should be linear functions that go through alloy A and B. 
% The intersection of the line is evaluated at alloy A.
% The axis_element is set to Alloy A composition.
for index = 1:length(elements)
	element = elements(index);
    if element == axis_element
        condition = "W(" + element + ")";
        conditions(condition) = alloy_A_input(element);
    else
        element_1 = element;
        element_2 = elements(index + 1);
        slope = (alloy_B(element_1) - alloy_A(element_1)) / ...
            (alloy_B(element_2) - alloy_A(element_2));
        m = alloy_A(element_1) - slope * alloy_A(element_2);
        condition = "W(" + element_1 + ")" +sprintf('%+0.5f',-slope) + ...
            "*W(" + element_2 + ")";
        conditions(condition) = m;
    end
end
session = tc_toolbox.TCToolbox();

[filepath,name,ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder( name + "_cache");

elements = [dependent_element elements_input];
database = "FEDEMO";
tc_system = session.select_database_and_elements(database, elements)...
                     .get_system();

axis_condition = "W(" + axis_element + ")";
calculation_step = tc_system.with_property_diagram_calculation()...
    .with_axis(tc_toolbox.step_or_map_diagrams.CalculationAxis(axis_condition)...
        .set_min(alloy_A(axis_element))...
        .set_max(alloy_B(axis_element))...
        .set_start_at(alloy_A(axis_element)));
for condition_obj = conditions.keys()
    condition = condition_obj{1};
    disp([condition '=' sprintf("%0.5f", conditions(condition))])
    calculation_step.set_condition(condition, conditions(condition));
end


calculated_results_step = calculation_step.calculate();
plot_data_step = calculated_results_step.get_values_grouped_by_quantity_of(...
    "W(" + axis_element + ")", "VPV(*)");
 
calculation_map = tc_system.with_phase_diagram_calculation()...
    .with_first_axis(...
        tc_toolbox.step_or_map_diagrams.CalculationAxis(axis_condition)...
            .set_min(alloy_A(axis_element))...
            .set_max(alloy_B(axis_element))...
            .set_start_at(alloy_A(axis_element))...
    ).with_second_axis(...
    	tc_toolbox.step_or_map_diagrams.CalculationAxis("T")...
            .set_min(500)...
            .set_max(3000.0)...
    );

for condition_obj = conditions.keys()
    condition = condition_obj{1};
    disp([condition "=" sprintf("%0.5f", conditions(condition))]);
    calculation_map.set_condition(condition, conditions(condition));
end

calculated_results_map = calculation_map.calculate();

plot_data_phase_diagram = calculated_results_map...
    .get_values_grouped_by_stable_phases_of(...
        tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(axis_element),...
        tc_toolbox.ThermodynamicQuantity.temperature());
    
plot_data_step_compositions = calculated_results_step...
    .get_values_grouped_by_quantity_of(...
        tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(axis_element),...
        tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(...
            tc_toolbox.Constants.ALL_PHASES));


figure()
sgtitle("Fe-Cr-Ni-C-Mn: Alloy A to Alloy B")

add_data_to_axes(1, plot_data_step, "Property diagram T=1000K", ...
    axis_condition, "Volume fraction phases [-]");

add_data_to_axes(2, plot_data_phase_diagram.get_lines(), "Isopleth", ...
    axis_condition, "T [K]");
add_data_to_axes(3, plot_data_step_compositions, ...
    "Linear composition conditions", axis_condition, ...
    "Weight-fraction elements [-]");



function add_data_to_axes(plot_index, plot_data, sub_title, x_label, y_label)
    subplot(3,1,plot_index);
    title(sub_title)
    xlabel(x_label)
    ylabel(y_label)

    hold on                                          
    for k = plot_data.keys()
        label = k{1};  
        group = plot_data(label);
        plot(group.get_x, group.get_y, 'DisplayName', group.get_label, ...
            'LineWidth', 2)
    end
    hold off
    legend
    leg = findobj(gcf, 'Type', 'Legend');
    set(leg,'Interpreter', 'none') 
    axis tight
end
