%{

This example simulates the kinetics of precipitation of CEMENTITE from a BCC_A2 solution phase and shows the grain growth results
for two cases: with and without Zener pining. A log normal distribution is applied for the matrix grain size.

%}

cut_off = 1e-9;
mu_log_norm = 2.0E-6;
sigma_log_norm = 0.5;
sample_numbers=1000;

sim_time = 126000;

[filepath,name,ext] = fileparts(mfilename('fullpath'));

session = tc_toolbox.TCToolbox();
session.set_cache_folder(name + "_cache");

system_builder = session.select_thermodynamic_and_kinetic_databases_with_elements("FEDEMO", "MFEDEMO", ["Fe", "C"]);
sys = system_builder.get_system();
calculation = sys.with_isothermal_precipitation_calculation();
calculation.set_composition_unit(tc_toolbox.CompositionUnit.MASS_PERCENT);
calculation.set_composition("C",0.2);
calculation.set_temperature(995.15);
calculation.set_simulation_time(sim_time);

grain_size_distribution= get_log_normal_distribution(cut_off,mu_log_norm,sigma_log_norm,sample_numbers);
no_pinning_results = simulate(calculation,grain_size_distribution,false);
with_pinning_results = simulate(calculation,grain_size_distribution,true);
plot_results(no_pinning_results,with_pinning_results);

function grain_size_distribution= get_log_normal_distribution(cut_off,mu,sigma,sample_numbers)
grain_size_distribution = tc_toolbox.precipitation.GrainSizeDistribution();
radius=zeros(0,sample_numbers);
density=zeros(0,sample_numbers);
step =(5*mu-cut_off)/sample_numbers;
for i =1:sample_numbers
    radius(i)=cut_off +i*step;
    density(i)=(1/mu)*(1/((radius(i)/mu)*sigma*sqrt(2*pi)))*exp(-(log(radius(i)/mu)-mu)^2/(2*sigma^2));
    grain_size_distribution.add_radius_and_number_density(radius(i),density(i));
end
end

function sim_results = simulate(system,grain_size_distribution,hasPinner)
grain_growth = tc_toolbox.precipitation.GrainGrowth(grain_size_distribution);
grain_growth.set_grain_boundary_mobility_activation_energy(242000.0);
grain_growth.set_grain_boundary_energy(0.5);
grain_growth.set_grain_boundary_mobility_pre_factor(0.004);
if hasPinner
    grain_growth.enable_zener_pinning();
else
    grain_growth.disable_zener_pinning();
end
matrixphase = tc_toolbox.precipitation.MatrixPhase("BCC_A2")...
    .with_grain_growth_model(grain_growth)...
    .add_precipitate_phase(...
    tc_toolbox.precipitation.PrecipitatePhase("CEMENTITE") ...
    .set_nucleation_in_bulk() ...
    .set_interfacial_energy_estimation_prefactor(0.15));

calculation = system.with_matrix_phase(matrixphase);

sim_results = calculation.calculate();
end

function plot_results(no_pinning_results,with_pinning_results)
exp_one_x = [0.167, 0.583, 2.5, 4.5, 7.0, 11.0, 21.0, 35.0];
exp_one_y = [3.3, 3.6, 4.4, 4.8, 5.7, 5.4, 6.7, 7.1];
[time_no_pining, mean_grain_size_no_pining] = no_pinning_results.get_grain_mean_radius();
[time_with_pining, mean_grain_size_with_pining] = with_pinning_results.get_grain_mean_radius();
scatter(exp_one_x,exp_one_y,'^')
hold on
plot(time_no_pining/3600,mean_grain_size_no_pining*1e6);
hold on
plot(time_with_pining/3600,mean_grain_size_with_pining*1e6,'g');
hold off
legend('Experiment','Mean radius of BCC A2 (No pinning)','Mean radius of BCC A2 (With pinning)');
xlabel('Time [h]');
ylabel('Length [μm]');
end
