%{
This example models the homogenization of a cast Ni-Cr steel and compares predictions with the experimental data
collated by Fuchs and Roósz (1).

As a first step, the microsegregation after solidification is simulated using a Scheil-calculation, then
as a second step the homogenization is simulated using a diffusion simulation.

The conditions studied have a dendritic half spacing of 200 µm and are heat treated at 1120 °C.
The data has been de-normalized to allow comparison using weight percent composition values.
The composition of the steel is 0.4C, 0.65Mn, 0.35Si, 0.015S, 0.01P, 1.9Ni, 0.95Cr and 0.3 Mo (wt.%).
The diffusion of Cr and Ni are modelled using a simplified chemistry of 0.4C, 0.65Mn, 1.9Ni, and 0.95Cr using the
FEDEMO thermodynamic and MFEDEMO mobility databases.

References
1. Homogenization of Iron-Base Cast Alloys. Fuchs, E. G., and A. Roósz. 1975, Metal Science, pp. 111-118.
%}

thermodynamic_database = "FEDEMO";
mobility_database = "MFEDEMO";
dependent_element = "Fe";
composition_wt_pct = containers.Map(["Ni", "Cr", "C", "Mn"], {1.9, 0.95, 0.4, 0.65}); % in wt-%
fast_diffuser = "C";
num_interp_points = 100;
alloy = "Fuchs and Roosz (1975)";
dendrite_arm_spacing = 400; % in micrometers
homogenization_temperature = 1120; % in degree C
homogenization_time = 72.0; % in hours
FIRST_TIMEPOINT_WITH_PHASES = 1e-5; % to get the initial phase distribution close to t=0

session = tc_toolbox.TCToolbox();
elements = composition_wt_pct.keys();
system_builder = session.select_thermodynamic_and_kinetic_databases_with_elements(thermodynamic_database, mobility_database, ...
    cat(2, elements, dependent_element));
system = system_builder.get_system();

calc = system.with_scheil_calculation();
calc.with_calculation_type(tc_toolbox.scheil.ScheilCalculationType.scheil_classic().set_fast_diffusing_elements([fast_diffuser]));
calc.set_composition_unit(tc_toolbox.CompositionUnit.MASS_PERCENT);

for key = composition_wt_pct.keys()
    element = key{1};
    calc.set_composition(element, composition_wt_pct(element));
end

result = calc.calculate();

% obtain all phases formed in the system
scheil_curve = result.get_values_grouped_by_stable_phases_of(tc_toolbox.ScheilQuantity.mole_fraction_of_all_solid_phases(), ...
    tc_toolbox.ScheilQuantity.temperature());

phases_in_calc = result.get_stable_phases();

% get the solid concentration along the solidification
average_solid_compo_in_slice = containers.Map();
for key_element = elements
    element = key_element{1};
    [total_solid_fracs, average_solid_compo_in_slice_this_element] = result.get_values_of(...
        tc_toolbox.ScheilQuantity.mole_fraction_of_all_solid_phases(), ...
        tc_toolbox.ScheilQuantity.average_composition_of_solid_phases_as_mass_fraction(element));
    average_solid_compo_in_slice(element) = average_solid_compo_in_slice_this_element;
end

% interpolate to new grid
sec_dendrite_arm_spacing = dendrite_arm_spacing * 1e-6 / 2;
grid_in_dendrite_from_scheil = total_solid_fracs * sec_dendrite_arm_spacing;
grid_in_dendrite_interpolated = linspace(0, sec_dendrite_arm_spacing, num_interp_points);
average_solid_compo_in_slice_interpolated = interpolate_scheil_composition_profiles(...
    grid_in_dendrite_from_scheil, average_solid_compo_in_slice, grid_in_dendrite_interpolated, elements);

% composition profiles for fast diffusers are not available, use average alloy composition instead
for element = elements
    if element{1} == fast_diffuser
        average_solid_compo_in_slice_interpolated(element{1}) = ...
            ones(size(grid_in_dendrite_interpolated)) * composition_wt_pct(element{1}) * 1e-2;
    end
end

concentration_profiles = tc_toolbox.diffusion.PointByPointGrid(tc_toolbox.diffusion.Unit.MASS_FRACTION);
for i = 1:num_interp_points
    gridpoint = tc_toolbox.diffusion.GridPoint(grid_in_dendrite_interpolated(i));
    for element = elements
        compo = average_solid_compo_in_slice_interpolated(element{1});
        gridpoint.add_composition(element{1}, compo(i));
    end
    concentration_profiles.add_point(gridpoint);
end

dendrite_region = tc_toolbox.diffusion.Region("Solid").with_point_by_point_grid_containing_compositions(concentration_profiles);

for phase = phases_in_calc.transpose()
    if ~system.get_phase_object(phase{1}).is_liquid()
        dendrite_region.add_phase(phase{1});
    end
end

dictra = system.with_isothermal_diffusion_calculation();
dictra.set_temperature(homogenization_temperature + 273.15);
dictra.set_simulation_time(homogenization_time * 3600);
dictra.add_region(dendrite_region);
dictra.with_solver(tc_toolbox.diffusion.Solver.automatic());

homo_results = dictra.calculate();

% plot results
figure(1)
for key_label = scheil_curve.keys()
    label = key_label{1};
    scheil_stable_phases = scheil_curve(label);
    plot(scheil_stable_phases.get_x(), scheil_stable_phases.get_y(), "DisplayName", label);
    hold on
end
title(alloy + ", Dendrite arm spacing = " + dendrite_arm_spacing + " {\mu}m, ");
xlabel("Fraction solid phase");
ylabel("Temperature (K)")
legend("Interpreter", "none")
hold off

figure(2)
set(gca, "linestyleorder",["-", "--", "-."], "colororder", [0.83 0.14 0.14; 1.00 0.54 0.00; 0.47 0.25 0.80; 0.25 0.80 0.54], "nextplot", "add");
for time = [FIRST_TIMEPOINT_WITH_PHASES, homogenization_time / 2, homogenization_time]
    for key_element = elements
        element = key_element{1};
        [distance, w_solid_homogenized] = homo_results.get_mass_fraction_of_component_at_time(element, time * 3600);
        if time == FIRST_TIMEPOINT_WITH_PHASES
            time_label = 0;
        else
            time_label = time;
        end
        plot(distance .* 1.0e6, w_solid_homogenized * 100, "DisplayName", element + " after " + time_label + " h");
        hold on
    end
end

title(alloy + ", Dendrite arm spacing = " + dendrite_arm_spacing + " {\mu}m, " + homogenization_temperature + " {\circ}C");
xlabel("distance ({\mu}m)");
ylabel("concentration (mass-percent)");
legend("Interpreter", "none");
hold off

figure(3)
% Experimental data from Fuchs and Roósz (1975)
cr_1120C_exp = [1.4924789, 1.3224482, 1.1038373];
ni_1120C_exp = [2.8543548, 2.8041256, 2.4525212];
time_exp_h = [8.0, 24, 72];

[time, w_f_cr] = homo_results.get_mass_fraction_at_upper_interface("Solid", "Cr");
plot(time / 3600, w_f_cr * 100, "red", "DisplayName", "Cr");
hold on;
scatter(time_exp_h, cr_1120C_exp, "red", "filled", "square");

[time, w_f_ni] = homo_results.get_mass_fraction_at_upper_interface("Solid", "Ni");
plot(time / 3600, w_f_ni * 100, "blue", "DisplayName", "Ni");
scatter(time_exp_h, ni_1120C_exp, "blue", "filled", "o");

h=get(gca,"Children"); % grab all the axes handles at once
legendstr={'Cr', 'Cr', 'Ni', 'Ni'};
legend(h([1 3]),legendstr{[1 3]})

title(alloy + ", Dendrite arm spacing = " + dendrite_arm_spacing + " {\mu}m, " + homogenization_temperature + " {\circ}C");
xlabel("time (hours)");
ylabel("Concentration (mass-percent)");
legend("Interpreter", "none");
ylim([0, 4])
hold off

function interpolated_composition = interpolate_scheil_composition_profiles(grid_original, composition_original, grid_interpolated, elements)
%     The constant composition_wt_pct during the Scheil-steps should be kept also when interpolating to a new grid.
%     Otherwise, it could be errors in the mass balance.
%     E.g. the mass balance is rather off if one interpolates with a linear function between previous and final Scheil
%     step in a case that a large fraction solidifies with eutectic composition_wt_pct at the last Scheil step.

    interpolated_composition = containers.Map('KeyType', 'char', 'ValueType', 'any');
    for element = elements
        interpolated_composition(element{1}) = ones(size(grid_interpolated));
    end

    i_interp = 1;
    for gp_interp = grid_interpolated
        i_orig = 1;
        for gp_orig = transpose(grid_original)
            if gp_interp < grid_original(1)
                % use composition_wt_pct of first Scheil step
                for element = elements
                    element_composition = composition_original(element{1});
                    element_interpolated_comp = interpolated_composition(element{1});
                    element_interpolated_comp(i_interp) = element_composition(i_orig);
                    interpolated_composition(element{1}) = element_interpolated_comp;
                end
                break;
            elseif gp_interp > grid_original(end)
                % use composition_wt_pct of last Scheil step
                for element = elements
                   element_composition = composition_original(element{1});
                   element_interpolated_comp = interpolated_composition(element{1});
                   element_interpolated_comp(i_interp) = element_composition(end);
                   interpolated_composition(element{1}) = element_interpolated_comp;
                end
                break;
            elseif gp_interp >= gp_orig && gp_interp < grid_original(i_orig + 1)
                % use the Scheil-step composition_wt_pct for intermediate values
                for element = elements
                   element_composition = composition_original(element{1});
                   element_interpolated_comp = interpolated_composition(element{1});
                   element_interpolated_comp(i_interp) = element_composition(i_orig + 1);
                   interpolated_composition(element{1}) = element_interpolated_comp;
                end
                break;
            end
            i_orig = i_orig + 1;
        end
        i_interp = i_interp + 1;
    end
end
