%{
In this isothermal calculation example, the precipitation of the Cu4Ti phase in a Cu-Ti binary alloy is calculated.
To make a comparison, two separate simulations are performed:
(1) one assuming spherical morphology without elastic strain energy, and
(2) one assuming needle morphology whose shape, determined by competition between interfacial energy and elastic
    strain energy, is changed during the simulation.
%}

session = tc_toolbox.TCToolbox();

[filepath, name, ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder(name + "_cache");

tc_system = session...
    .select_thermodynamic_and_kinetic_databases_with_elements(...
       "CUDEMO", "MCUDEMO", ["Cu", "Ti"])...
    .select_phase("FCC_L12")...
    .select_phase("CU4TI1")...
    .get_system();

precipitation = tc_system....
    .with_isothermal_precipitation_calculation()...
       .set_composition_unit(tc_toolbox.CompositionUnit.MOLE_PERCENT)...
       .set_composition("Ti", 1.9)...
       .set_temperature(623.15)...
       .set_simulation_time(1.e5);
          
precip_sphere = tc_toolbox.precipitation.PrecipitatePhase("CU4TI1");

precip_needle = tc_toolbox.precipitation.PrecipitatePhase("CU4TI1")...
     .set_precipitate_morphology(...
        tc_toolbox.precipitation.PrecipitateMorphology.NEEDLE)...
     .enable_calculate_aspect_ratio_from_elastic_energy()...
     .set_transformation_strain_calculation_option(...
        tc_toolbox.precipitation....
          .TransformationStrainCalculationOption.USER_DEFINED)...
     .with_elastic_properties(tc_toolbox.precipitation...
                                 .PrecipitateElasticProperties()...
                                    .set_e11(0.022)...
                                    .set_e22(0.022)...
                                    .set_e33(0.003)...
                              );
                 
matrix_w_sphere = tc_toolbox.precipitation.MatrixPhase("FCC_L12")...
                     .set_mobility_adjustment('all', 100.0, 0.0)...
                     .add_precipitate_phase(precip_sphere);
               
matrix_w_needle = tc_toolbox.precipitation.MatrixPhase("FCC_L12")...
                     .set_mobility_adjustment('all', 100.0, 0.0)...
                     .with_elastic_properties_cubic(168.4, 121.4, 75.4)...
                     .add_precipitate_phase(precip_needle);

result_sphere = precipitation.with_matrix_phase(matrix_w_sphere).calculate();
result_needle = precipitation.with_matrix_phase(matrix_w_needle).calculate();

[mean_aspect_ratio_2_time, mean_aspect_ratio_2] = ...
    result_needle.get_mean_aspect_ratio_of("CU4TI1");

[number_density_1_time, number_density_1] = ...
    result_sphere.get_number_density_of("CU4TI1");
[number_density_2_time, number_density_2] = ...
    result_needle.get_number_density_of("CU4TI1");

[f_1_time, f_1] = ...
    result_needle.get_size_distribution_for_radius_of("CU4TI1",1e5);
[f_2_time, f_2] = ...
    result_needle.get_aspect_ratio_distribution_for_radius_of("CU4TI1", 1e5);

[mean_radius_1_time, mean_radius_1] = ...
    result_sphere.get_mean_radius_of("CU4TI1");
[mean_radius_2_time, mean_radius_2] = ...
    result_needle.get_mean_radius_of("CU4TI1");

figure()
semilogx(mean_aspect_ratio_2_time, mean_aspect_ratio_2, "DisplayName",...
    "Mean aspect ratio of CU4TI1 (Bulk) (Needle)");
xlabel("Time [s]");
ylabel("Mean aspect ratio");
legend;

figure()
loglog(number_density_1_time, number_density_1, "DisplayName",...
    "Number density of CU4TI1 (Bulk) (Sphere)");
hold on
loglog(number_density_2_time, number_density_2, "DisplayName",...
    "Number density of CU4TI1 (Bulk) (Needle)");
legend;
xlabel("Time [s]");
ylabel("Number density [m^-3]");
hold off

figure()
yyaxis left
plot(f_1_time, f_1, "DisplayName",...
    "Size distribution of CU4TI1 (Bulk) (Needle)");
xlabel("Length [m]");
ylabel("Size distribution [m^-4]");
yyaxis right
plot(f_2_time, f_2, "DisplayName",...
    "Aspect ratio distribution of CU4TI1 (Bulk) (Needle)");
ylabel("Aspect ratio distribution")
legend;

figure()
loglog(mean_radius_1_time, mean_radius_1, "DisplayName",...
    "Mean radius of CU4TI1 (Bulk) (Sphere)");
hold on
loglog(mean_radius_2_time, mean_radius_2, "DisplayName",...
    "Mean radius of CU4TI1 (Bulk) (Needle)");
xlabel("Time [s]");
ylabel("Length [m]");
legend;
hold off
