%{
This example simulates the kinetics of precipitation of both stable 
and metastable carbides from ferrite phase. It demonstrates that metastable 
carbides (cementite and M7C3) may first emerge and then disappear and the 
stable phase(M23C6) prevails.
%}

import tc_toolbox.precipitation.*;

session = tc_toolbox.TCToolbox();

[filepath, name, ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder(name + "_cache");

tc_system = session...
               .select_thermodynamic_and_kinetic_databases_with_elements(...
                  "FEDEMO", "MFEDEMO", ["Fe", "C", "Cr"]).get_system();


precipitation = tc_system.with_isothermal_precipitation_calculation();
precipitation.set_composition_unit(tc_toolbox.CompositionUnit.MASS_PERCENT);
precipitation.set_composition("C", 0.1);
precipitation.set_composition("Cr", 12);

cementite_phase = tc_toolbox.precipitation.PrecipitatePhase("CEMENTITE")...
                     .set_interfacial_energy(0.167)...
                     .set_nucleation_at_grain_boundaries();
                          
m7c3_phase = tc_toolbox.precipitation.PrecipitatePhase("M7C3")...
                .set_interfacial_energy(0.282)...
                .set_nucleation_at_grain_boundaries();
                          
m23c6_phase = tc_toolbox.precipitation.PrecipitatePhase("M23C6")...
                 .set_interfacial_energy(0.252)...
                 .set_nucleation_at_grain_boundaries();

precipitation...
    .with_matrix_phase(tc_toolbox.precipitation.MatrixPhase("BCC_A2")...
                          .with_grain_growth_model(GrainGrowthModel.fixed_grain_size(1.0E-4))...
    .add_precipitate_phase(cementite_phase)...
    .add_precipitate_phase(m7c3_phase)...
    .add_precipitate_phase(m23c6_phase));

precipitation.set_temperature(1053)...
    .set_simulation_time(4e5);

results = precipitation.calculate();

[time_cementite, volume_fraction_of_cementite] = ...
    results.get_volume_fraction_of("CEMENTITE");

[time_M7C3, volume_fraction_of_M7C3] = ...
    results.get_volume_fraction_of("M7C3");

[time_M23C6, volume_fraction_of_M23C6] = ...
    results.get_volume_fraction_of("M23C6");

figure();
hold off;
title("Three carbides precipitation");
xlabel("Time [s]");
ylabel("Volume Fraction");

semilogx(time_cementite, volume_fraction_of_cementite, "DisplayName",...
          "CEMENTITE (Grain boundaries)");
hold on;
semilogx(time_M7C3, volume_fraction_of_M7C3, "DisplayName",...
    "M7C3 (Grain boundaries)");

semilogx(time_M23C6, volume_fraction_of_M23C6, "DisplayName",...
    "M23C6 (Grain boundaries)");

legend;


