%{
Calculates the phases occurring for a material mixture, in this case for a
mixture of a martensitic stainless steel with Alloy 800.

This type of calculation is for example useful for understanding effects
when welding dissimilar materials - without the need to perform diffusion
calculations.
%}
import tc_toolbox.material_to_material.*;

session = tc_toolbox.TCToolbox();

independent_elements = ["Cr", "Ni"];
material_a_composition = [17.0, 2.0];
material_b_composition = [19.0, 35.0];

[filepath,name,ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder( name + "_cache");

sb = session.select_database_and_elements("FEDEMO", ["Fe", "Cr", "Ni"]);
tc_system = sb.get_system();

calc = tc_system.with_material_to_material().with_single_equilibrium_calculation();
calc.set_material_a(containers.Map(independent_elements, material_a_composition), "Fe");
calc.set_material_b(containers.Map(independent_elements, material_b_composition), "Fe");
calc.set_composition_unit(tc_toolbox.CompositionUnit.MASS_PERCENT);
calc.with_first_constant_condition(ConstantCondition.fraction_of_material_b(0.4));
calc.with_second_constant_condition(ConstantCondition.temperature(650 + 273.15));
result = calc.calculate();

disp("Stable phases at 650 degree C, 40 wt-% Alloy 800:")
disp(result.get_stable_phases())
disp("Molar fractions:")
list_stable_phases(result)

function list_stable_phases(result)
    stable_phases = result.get_stable_phases();
    for index = 1: length(stable_phases)
        phase = stable_phases{index};
        disp("Amount of " + phase + " = " + ...
            100 * result.get_value_of("NP(" + phase + ")") + "%");
    end
end