%{
Calculates the microsegregation profile in a solidified Ni-based superalloy 
using the Scheil-method. The plot is normalized relative to the nominal 
composition and shows the composition of the matrix phase (FCC).

A model alloy in the system Ni-Al-Cr is used as an example.
%}
database = "NIDEMO";
dependent_element = "Ni";
composition = containers.Map({'Al', 'Cr'}, ... % in wt-%
                             {7.0 , 7.0});
relevant_phase = "FCC_L12";
elements = string(composition.keys());

session = tc_toolbox.TCToolbox();

[filepath,name,ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder( name + "_cache");

tc_system = session.select_database_and_elements(database, ...
    [dependent_element elements])...
    .get_system();


% convert the nominal composition of the alloy to atomic fractions
% takes elements in camel-case
atomic_frac = tc_system.convert_composition(composition, ...
    tc_toolbox.ConversionUnit.WEIGHT_PERCENT, ...
    tc_toolbox.ConversionUnit.MOLE_FRACTION,...
    dependent_element);
                             
% perform the Scheil calculation for the same system
scheil_calculator = tc_system...
                      .with_scheil_calculation()...
                      .set_composition_unit(...
                        tc_toolbox.CompositionUnit.MASS_PERCENT);

for element = elements
    scheil_calculator.set_composition(element, composition(element));
end

scheil_result = scheil_calculator.calculate();

scheil_result_data = ...
    scheil_result.get_values_grouped_by_quantity_of(...
        tc_toolbox.ScheilQuantity.mole_fraction_of_all_solid_phases(),...
        tc_toolbox.ScheilQuantity.composition_of_phase_as_mole_fraction(...
            relevant_phase, "All"));
figure()
sgtitle("Microsegregation in NiAl7Cr7")
xlabel("Percentage of solidified material [-]")
ylabel("Normalized composition of the FCC phase [-]")

hold on   

for k = scheil_result_data.keys()
    label = k{1};  
    % the format of the label is (here) always "EL in PHASE"
    splitted_label = split(label);
    element = to_camelcase(splitted_label{1});
    if ~strcmpi(element, dependent_element)
        section = scheil_result_data(label);
        total_solid_frac = section.get_x;
        element_frac_composition = section.get_y;
        normalized_element_composition = ...
            element_frac_composition ./ atomic_frac(element);
        plot(100 * total_solid_frac, normalized_element_composition, ...
            "DisplayName", element, "LineWidth", 2)
    end 
end                                             
legend

function [str_camel] = to_camelcase(str)
    str_camel = regexprep(lower(str),"(\<[a-z])", "${upper($1)}");
end
