%{
Calculating Interfacial Energy with a Property Model

This example shows how to run a property model from TC-Python to calculate the interfacial 
energy. The Thermo-Calc Property Model uses a broken-bond approach. Any other pre-installed or 
user-defined Property Model created in Thermo-Calc can be loaded and used in the same way as in this 
example. Search the Thermo-Calc Help (in Thermo-Calc press F1) for more information about the Property
Models and the Property Model Development Framework. 


The details of a property model (arguments, result quantities, and so forth) are defined in the Thermo-Calc Property 
Model and are not known to TC-Toolbox in advance. Therefore, when using property models, the 
workflow differs compared to other TC-Toolbox modules. The available arguments need to be queried by using 
the respective `get_...()`- method. This requires some interactive development style until a first working
version of the code is set up.
%}

clc

session = tc_toolbox.TCToolbox();
temp = 800;  % in degree C
system_builder = session.select_database_and_elements("NIDEMO", ["Ni","Al","Cr"])...
    .without_default_phases()...
        .select_phase("FCC_L12#1").select_phase("FCC_L12#2");
tc_system = system_builder.get_system();

propertyModelCalculation = tc_system.with_property_model_calculation("Interfacial energy");
propertyModelCalculation.set_temperature(temp + 273.15);
propertyModelCalculation.set_composition_unit(tc_toolbox.CompositionUnit.MASS_PERCENT);
propertyModelCalculation.set_composition("Al", 5.0);
propertyModelCalculation.set_composition("Cr", 10.0);

% asking for the available arguments, if we do not change them, they will be set to the default defined in the model
args = propertyModelCalculation.get_arguments();
disp(["Available arguments: ", transpose(args)]);

propertyModelCalculation.set_argument("matrix", "FCC_L12");
propertyModelCalculation.set_argument("precipitate", "FCC_L12#2");

result = propertyModelCalculation.calculate();

% asking for the available result quantities
disp(["Available result quantities: ", result.get_result_quantities()]);
disp(["Description of interfacialEnergy: ", result.get_result_quantity_description("interfacialEnergy")]);

interfacial_energy = result.get_value_of("interfacialEnergy");
disp(["Interfacial energy at ", temp, char(176)+"C ", 1000 * interfacial_energy, "mJ/m**2"]);


