%{
This example shows how to create adiabatic equilibrium calculations for
varied pressure. The equilibrium between a steel and slag melt
and oxygen is calculated.            
%}

import tc_toolbox.process_metallurgy.base.*;
import tc_toolbox.process_metallurgy.equilibrium.*;
import tc_toolbox.*

session = tc_toolbox.TCToolbox();

[filepath, name, ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder(name + "_cache");

temp_in_c = 1700;
min_pressure_in_bar = 1.0;
max_pressure_in_bar = 2.0;
pressure_step_in_bar = 0.1;

calc = session.with_metallurgy().with_adiabatic_equilibrium_calculation(ProcessDatabase.OXDEMO);

steel = EquilibriumAddition(containers.Map(["Fe", "C"], {NaN, 4}), 100.0e3, temp_in_c + 273.15, CompositionUnit.MASS_PERCENT);
slag = EquilibriumAddition(containers.Map(["CaO", "SiO2"], {70, 30}), 3.0e3, temp_in_c + 273.15, CompositionUnit.MASS_PERCENT);
gas = EquilibriumGasAddition(containers.Map({'O2'}, {100}), 20 + 273.15, 1000, GasAmountUnit.NORM_CUBIC_METER, GasCompositionUnit.MASS_PERCENT);

(calc...
 .add_addition(steel)...
 .add_addition(slag)...
 .add_addition(gas));

pressure_in_bar = min_pressure_in_bar;
num_steps = 1 + int32((max_pressure_in_bar - min_pressure_in_bar) / pressure_step_in_bar);

pressure_range_in_bar = zeros(1, num_steps);
c_in_steel = zeros(1, num_steps);
slag_property = zeros(1, num_steps);

for m = 0:num_steps-1
    pressure_in_bar = min_pressure_in_bar + double(m) * pressure_step_in_bar;
    calc.set_pressure(pressure_in_bar * 1e5);
    result = calc.calculate();
    
    comp = result.get_composition_of_phase_group(PhaseGroup.ALL_METAL);
    c_in_steel(1, m+1) = comp('C');
    slag_property(1, m+1) = result.get_slag_property(SlagProperty.B2, SlagType.ALL);
    pressure_range_in_bar(1, m+1) = pressure_in_bar;
    disp(strcat('pressure_in_bar: ', num2str(pressure_in_bar)));
    disp(strcat('Stable phases: ' , transpose(result.get_stable_phases())));
end

figure(1);
plot(pressure_range_in_bar, c_in_steel);
title("100 t Fe-4 wt-% C / 3 t 70 wt-% CaO-30 wt-% SiO2 / 1000 Nm**3 O2");
xlabel('pressure / bar');
ylabel('C-content in steel / wt-%');

figure(2);
plot(pressure_range_in_bar, slag_property);
title("Slag property B2 (CaO / SiO2)");
xlabel('pressure / bar');
ylabel('slag property');

