%{
This example shows how to create an adiabatic equilibrium calculation for a
process metallurgical problem. The equilibrium between a steel and slag melt
and oxygen is calculated.            
%}

import tc_toolbox.process_metallurgy.base.*;
import tc_toolbox.process_metallurgy.equilibrium.*;
import tc_toolbox.*

session = tc_toolbox.TCToolbox();

[filepath, name, ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder(name + "_cache");

database = ProcessDatabase.OXDEMO;
metal_composition = containers.Map(["Fe", "C", "Si"], {NaN, 4.5, 1.0});  % in mass percent
total_metal_mass_in_t = 100;
slag_composition = containers.Map(["CaO", "Al2O3"], {75, 25}); % in mass percent
total_slag_mass_in_t = 3;
gas_composition = containers.Map({'O2'}, {100});  % in wt-%
total_gas_volume = 600;  % in nm**3
temp_in_c = 1650;  % in degree C

metal = EquilibriumAddition(metal_composition,...
                            1e3 * total_metal_mass_in_t,...
                            temp_in_c + 273.15, CompositionUnit.MASS_PERCENT);
slag = EquilibriumAddition(slag_composition,...
                           1e3 * total_slag_mass_in_t,...
                           temp_in_c + 273.15, CompositionUnit.MASS_PERCENT);
gas = EquilibriumGasAddition(gas_composition,...
                             20 + 273.15,...
                             total_gas_volume, GasAmountUnit.NORM_CUBIC_METER, GasCompositionUnit.VOLUME_PERCENT);

calc = session.with_metallurgy().with_adiabatic_equilibrium_calculation(database);

(calc...
 .add_addition(metal)...
 .add_addition(slag)...
 .add_addition(gas));

result = calc.calculate();

disp(["Stable phases: ", transpose(result.get_stable_phases())]);
disp(strcat("temperature in C: ", num2str(result.get_value_of('T') -  273.15)));

disp(["Slag phases: ", transpose(result.get_stable_phases_in_phase_group(PhaseGroup.ALL_SLAG))]);

disp("Composition of the overall slag:")
slag_composition = result.get_composition_of_phase_group(PhaseGroup.ALL_SLAG, CompositionUnit.MOLE_PERCENT);
for components = keys(slag_composition)
    content = slag_composition(components{1});
    disp(strcat(components{1}, '=', num2str(content), ' mol-%'))
end


