%{
This example simulates the precipitation of gamma-prime phase from gamma 
phase in a Ni-Al-Cr alloy. It demonstrates how to run 
TTT(Temperature-Time-Transformation) and CCT(Continuous-Cooling-Transformation) 
diagrams. It also shows 
 (1) how to set up objects(the system, precipitate phase, and matrix phase) 
     and then reuse them, and
 (2) how to set an alias for the precipitate phase
%}

session = tc_toolbox.TCToolbox();

[filepath, name, ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder(name + "_cache");

tc_system = session...
    .select_thermodynamic_and_kinetic_databases_with_elements("NIDEMO",...
       "MNIDEMO", ["Ni", "Al", "Cr"])...
    .select_phase("DIS_FCC_A1")...
    .select_phase("FCC_L12#2")...
    .get_system();

precip = tc_toolbox.precipitation.PrecipitatePhase("FCC_L12#2")...
            .set_interfacial_energy(0.023).set_alias('GAMMA-PRIME');

matrix = tc_toolbox.precipitation.MatrixPhase("DIS_FCC_A1")...
            .add_precipitate_phase(precip);

ttt_calc = tc_system.with_ttt_precipitation_calculation();
ttt_calc.set_composition_unit(tc_toolbox.CompositionUnit.MOLE_PERCENT);
ttt_calc.set_composition("Al", 10);
ttt_calc.set_composition("Cr", 10);
ttt_calc.with_matrix_phase(matrix);
ttt_calc.set_min_temperature(1000);
ttt_calc.set_max_temperature(1160);
ttt_calc.set_temperature_step(10);
ttt_calc.set_max_annealing_time(1.0e6);
ttt_calc.stop_at_volume_fraction_of_phase(1.e-4);
ttt_results = ttt_calc.calculate();
                   
cct_calc = tc_system.with_cct_precipitation_calculation();
cct_calc.set_composition_unit(tc_toolbox.CompositionUnit.MOLE_PERCENT);
cct_calc.set_composition("Al", 10);
cct_calc.set_composition("Cr", 10);
cct_calc.with_matrix_phase(matrix);
cct_calc.set_min_temperature(1000);
cct_calc.set_max_temperature(1200);
cct_calc.set_cooling_rates([1., 1.e1, 1.e2, 1.e3]);
cct_calc.stop_at_volume_fraction_of_phase(1.e-4);
cct_results = cct_calc.calculate();

[ttt_time, ttt_temperature] = ...
    ttt_results.get_result_for_precipitate("GAMMA-PRIME");

[cct_time, cct_temperature] = ...
    cct_results.get_result_for_precipitate("GAMMA-PRIME");

figure();
hold off;
title("TTT and CCT");
xlabel("Time [s]");
ylabel("Temperature [K]");

semilogx(ttt_time, ttt_temperature, "DisplayName", "GAMMA-PRIME (TTT)");
hold on;
semilogx(cct_time, cct_temperature, "DisplayName", "GAMMA-PRIME (CCT)");
legend;

