%{
Calculates the variation of the 80% solid temperature using
Scheil-solidification for random compositions within a specification range.
The alloy system Al-Si-Cu is used as an example.
%}
num_samples = 40;  % should be higher for good statistics (for example N=100)
dependent_element = "Al";
database = "ALDEMO";
lower_spec_limit = containers.Map({'Si', 'Cu'}, ... % in wt-%
                                  {8.0 , 2.0});
upper_spec_limit = containers.Map({'Si', 'Cu'}, ... % in wt-%
                                  {11.0, 4.0});
required_solid_frac = 0.8;   % the temperature for which the solid fraction 
                             %    will be evaluated
start_temp = 650;            % in degree Celsius
final_solid_frac = 0.85;     % finishing condition of the calculation 
                             %    (needs be higher than the required fraction)
min_content = 1e-3;          % smallest element content accepted in the 
                             %    distribution (in wt-%)

[composition_samples, elements] = ...
    get_alloy_composition_distribution(...
        lower_spec_limit, upper_spec_limit, num_samples);
required_T_for_solid_frac = zeros(1, num_samples);

figure()
sgtitle("Variation of the solidification curves")
subplot(2, 1, 1);
xlabel("Temperature [°C]")
ylabel("Solid phase fraction [-]")
hold on   

session = tc_toolbox.TCToolbox();

[filepath,name,ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder( name + "_cache");

tc_system = session.select_database_and_elements(database, ...
    [dependent_element elements])...
        .get_system();

% caching is only meaningful for the system since each calculation will 
% have random composition
session.disable_caching();

scheil_calculation = tc_system.with_scheil_calculation()...
    .set_composition_unit(tc_toolbox.CompositionUnit.MASS_PERCENT)...
        .with_options(tc_toolbox.scheil.ScheilOptions()...
            .terminate_on_fraction_of_liquid_phase(1 - final_solid_frac))...
        .set_start_temperature(start_temp + 273.15)...
        .set_composition_unit(tc_toolbox.CompositionUnit.MASS_PERCENT);

for index_sample = 1 : num_samples
    for index_element = 1 : length(elements)
        element = elements(index_element);
        sample = composition_samples(element);
        content = sample(index_sample);

        % prevent negative concentrations generated by the distribution
        if content < min_content  % in wt-%
            content = min_content;
        end

        scheil_calculation.set_composition(element, content);
    end
    result = scheil_calculation.calculate();

    [mole_frac_sol, temperature] = ...
        result.get_values_of(...
            tc_toolbox.ScheilQuantity.mole_fraction_of_all_solid_phases(),...
            tc_toolbox.ScheilQuantity.temperature());
    
    % make sure to remove any duplicate entries before interpolation
    [mole_frac_sol_unique,ia,ic] = unique(mole_frac_sol);
    temperature_unique = temperature(ia);
    
    required_T_for_solid_frac(index_sample) = ...
        interp1(mole_frac_sol_unique, temperature_unique, required_solid_frac);

    plot(temperature_unique - 273.15, mole_frac_sol_unique);
end    

subplot(2, 1, 2);
xlabel("Temperature with solid fraction of " + required_solid_frac +"[°C]");
ylabel("Solid phase fraction [-]");
histogram(required_T_for_solid_frac - 273.15);


% calculate a set of compositions from the specification range assuming a 
% multivariate normal distribution.
function [composition_samples, elements] = ...
    get_alloy_composition_distribution(lower_spec_limit, ...
        upper_spec_limit, num_samples)
    elements = string(lower_spec_limit.keys());
    composition_samples = containers.Map();
    
    for index = 1 : length(elements)
        element = elements(index);
        mean_composition = ...
            (lower_spec_limit(element) + upper_spec_limit(element)) / 2;
        sigma = (upper_spec_limit(element) - lower_spec_limit(element)) / 4;
        composition_samples(element) = ...
            randn(1, num_samples) * sqrt(sigma) + mean_composition;
    end 
end
