%{
Shows the basic usage of Scheil-calculations and mixing them with 
equilibrium calculations.
%}
database = "ALDEMO";
dependent_element = "Al";
elements = "Si";
wt_pct_si = 1.0; % in wt-%
delta_temp = 1.0; % in K, only used for the equilibrium line

session = tc_toolbox.TCToolbox();

[filepath, name, ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder(name + "_cache");

tc_system = session.select_database_and_elements(database, ...
    [dependent_element, elements])...
        .get_system();

scheil_calculation = tc_system...
                      .with_scheil_calculation()...
                      .set_composition_unit(...
                            tc_toolbox.CompositionUnit.MASS_PERCENT)...
                      .set_composition("Si", wt_pct_si);

solidification = scheil_calculation.calculate();


% plot the solidification curve including the equilibrium
scheil_curve = solidification.get_values_grouped_by_stable_phases_of(...
    tc_toolbox.ScheilQuantity.mole_fraction_of_all_solid_phases(),...
    tc_toolbox.ScheilQuantity.temperature());

temp_min = 1e6;
temp_max = -1e6;

figure()
subplot(2,1,1);
sgtitle("Solidification of AlSi1");
xlabel("Mole fraction of all solid phases [-]");
ylabel("Temperature [°C]");

hold on   

for k = scheil_curve.keys()
    label = k{1};  
    section = scheil_curve(label);
    plot(section.get_x(), section.get_y() - 273.15, "DisplayName", ...
        section.get_label(), "LineWidth", 2)
    
    temp_min = min(min(section.get_y()), temp_min);
    temp_max = max(max(section.get_y()), temp_max);
end
xlim([0 1]);
ylim([temp_min temp_max] - 273.15);

% calculate the equilibrium solidification line (starting at liquidus)
prop_calculation = tc_system.with_property_diagram_calculation();
result = prop_calculation...
          .with_axis(tc_toolbox.step_or_map_diagrams.CalculationAxis(...
                tc_toolbox.ThermodynamicQuantity.temperature())...
                    .set_min(temp_min)...
                    .set_max(temp_max))...
          .set_condition("W(Si)", wt_pct_si / 100)...
          .calculate();
 
[temp_eq_frac, liquid_eq_frac] = ...
    result.get_values_of(...
        tc_toolbox.ThermodynamicQuantity.temperature(),...
        tc_toolbox.ThermodynamicQuantity.mole_fraction_of_a_phase("LIQUID"));

solid_eq_frac = 1 - liquid_eq_frac;
temp_eq_frac = temp_eq_frac - 273.15;
valid_indices = (solid_eq_frac < 1.0);  % cutting off values with 100% solid
 
plot(solid_eq_frac(valid_indices), temp_eq_frac(valid_indices), "--", ...
    "DisplayName", "Equilibrium");
leg1 = legend;
set(leg1, 'Interpreter', 'none');

% Plot the mole fraction of the solid phases separated for each
mole_frac_all_phases = ...
    solidification.get_values_grouped_by_stable_phases_of(...
        tc_toolbox.ScheilQuantity.mole_fraction_of_a_solid_phase(...
        tc_toolbox.Constants.ALL_PHASES),...
        tc_toolbox.ScheilQuantity.temperature());

subplot(2,1,2);
hold on
xlabel("Mole fraction of each solid phase [-]")
ylabel("Temperature [°C]")

for k = mole_frac_all_phases.keys()
    label = k{1};  
    section = mole_frac_all_phases(label);
    plot(section.get_x(), section.get_y() - 273.15, "DisplayName", ...
        section.get_label(), "LineWidth", 2);
end
hold off   

leg2 = legend;
set(leg2, "Interpreter", "none");
