%{
This script demonstrates how to call the Yield Strength model using TC-Python.
Starting from version 2023b, the Yield Strength model can be used in two different modes:
Simplified and Advanced, which is also demonstrated.

This example follows the GUI-example "PM_G_04_Yield_Strength"

The details of a property model (arguments, result quantities, and so forth) are defined in the Thermo-Calc Property 
Model and are not known to TC-Toolbox in advance. Therefore, when using property models, the 
workflow differs compared to other TC-Toolbox modules. The available arguments need to be queried by using 
the respective `get_...()`- method. This requires some interactive development style until a first working
version of the code is set up.
%}

clc
clf
close all;

import tc_toolbox.*

session = TCToolbox();
temp = 623;  % in K
min_r = 2e-10;
max_r = 3e-8;
no_of_steps = 50;
radii = linspace(min_r,max_r,no_of_steps);

% Define system and get data
system_builder = session.select_database_and_elements("ALDEMO", ["Al","Sc"]);
tc_system = system_builder.get_system();

% Set up the property model with conditions
propertyModelSetup = tc_system.with_property_model_calculation("Yield Strength")...
                    .set_temperature(temp)...
                    .set_composition_unit(tc_toolbox.CompositionUnit.MASS_PERCENT)...
                    .set_composition("Sc", 0.3);

% Set up the simplified mode calculation
simplifiedCalculation = propertyModelSetup...
                        .set_argument("Select mode", "Simplified")...
                        .set_argument("precip_str_selection_simplified_mode", "True")...
                        .set_argument("precipitate_simplified_mode-1", "AL3SC");

% Calculate for all radii in Simplified mode
simplified_result_vector = zeros(1,length(radii));
i = 1;
while i <= length(radii)
    simplifiedCalculation.set_argument("precipitate_radius_simplified_mode-1", radii(i));
    result_simple = simplifiedCalculation.calculate();
    total_precip_str_simplified = result_simple.get_value_of("Total precipitation strengthening")
    simplified_result_vector(i) = total_precip_str_simplified;
    i = i + 1;
end

% Set up the advanced mode calculation with a specific precipitation hardening model
advancedCalculation = propertyModelSetup...
                        .set_argument("Select mode", "Advanced")...
                        .set_argument("precip_str_selection", "True")...
                        .set_argument("Precipitation model list-1", "Deschamps model")...
                        .set_argument("Precipitate-1", "AL3SC");

% Calculate for all radii in Advanced mode
advanced_result_vector = zeros(1,length(radii));
i = 1;
while i <= length(radii)
    advancedCalculation.set_argument("Precipitate radius deschamps-1", radii(i));
    result_advanced = advancedCalculation.calculate();
    total_precip_str_advanced = result_advanced.get_value_of("Total precipitation strengthening")
    advanced_result_vector(i) = total_precip_str_advanced;
    i = i + 1;
end

% Plot results
plot(radii,simplified_result_vector,'o',radii,advanced_result_vector)
xlabel("Particle radius (m)")
ylabel("Total precipitation strengthening (MPa)")
legend("Simplified","Advanced")

