%{
Performance comparison for looping single equilibrium calculations using the
legacy toolbox, the new toolbox and batch equilibrium using the new
toolbox.

Legacy time:33.14s
New eq time:40.84s
New batch eq time:31.67s
%}


database = 'FEDEMO';
dependent_element = 'Fe';
composition = containers.Map({'Cr','Mn','Ni','C'}, ... % in wt-%
                             {1e-2 , 1e-2, 1e-2, 1e-3});
                         
solutes = string(composition.keys());
elements = [dependent_element solutes];
min_temperature = 500;
max_temperature = 1500;
nb_temperature_steps = 200;

timing_legacy = time_legacy_toolbox(database, composition, elements, solutes, min_temperature, max_temperature, nb_temperature_steps);
timing_new = time_new_toolbox(database, composition, elements, solutes, min_temperature, max_temperature, nb_temperature_steps);
timing_new_batch = time_new_toolbox_batch(database, composition, elements, solutes, min_temperature, max_temperature, nb_temperature_steps);

disp("Legacy time:" + num2str(timing_legacy) + "s")
disp("New eq time:" + num2str(timing_new) + "s")
disp("New batch eq time:" + num2str(timing_new_batch) + "s")

function [timing] = time_new_toolbox(database, composition, elements, solutes, min_temperature, max_temperature, nb_temperature_steps)
%     import tc_toolbox.*

    session = tc_toolbox.TCToolbox();
    session.set_ges_version(5);
    tc_system = session.select_database_and_elements(database, elements).get_system();
    calculator = tc_system.with_single_equilibrium_calculation();
    calculator.set_condition("T", 1200);
    for element = solutes
        calculator.set_condition(tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(element),...
                                 composition(element) / 100);
    end
    result = calculator.calculate();

    tic;
    temperatures = linspace(min_temperature, max_temperature, nb_temperature_steps);
    mu_c = zeros(1,length(temperatures));
    for i = 1:length(temperatures)
        calculator.set_condition("T", temperatures(i));
        result = calculator.calculate();
        mu_c(i) = result.get_value_of('mu(c)');
    end
    timing = toc;
    
    plot(temperatures, mu_c);
    xlabel("Temperature K");
    ylabel("Chemical potential C");

end


function [timing] = time_new_toolbox_batch(database, composition, elements, solutes, min_temperature, max_temperature, nb_temperature_steps)
%     import tc_toolbox.*

    session = tc_toolbox.TCToolbox();
    session.set_ges_version(5);
    tc_system = session.select_database_and_elements(database, elements).get_system();
    calculator = tc_system.with_batch_equilibrium_calculation();
    calculator.set_condition("T", max_temperature);
    for element = solutes
        calculator.set_condition(tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(element),...
                                 composition(element) / 100);
    end
    equilibria = {};
    
    tic;
    temperatures = linspace(min_temperature, max_temperature, nb_temperature_steps);
    mu_c = zeros(1,length(temperatures));
    for i = 1:length(temperatures)
        equilibria{i} = {{"T", temperatures(i)}};
    end    

    calculator.set_conditions_for_equilibria(equilibria);
    results = calculator.calculate(["mu(c)"], 200);
    mu_c = results.get_values_of("mu(c)");
    timing = toc;
    
    plot(temperatures, mu_c);
    xlabel("Temperature K");
    ylabel("Chemical potential C");
end


function [timing] = time_legacy_toolbox(database, composition, elements, solutes, min_temperature, max_temperature, nb_temperature_steps)
    tc_init_root;
    tc_open_database(database);
    check_error()

    selected_elements = convertStringsToChars(strjoin(elements));
    tc_element_select(selected_elements);
    tc_get_data;
    check_error()

    tc_set_condition('t', max_temperature);
    tc_set_condition('p', 101325);
    tc_set_condition('n', 1);

    for element = solutes
        condition = convertStringsToChars('x(' + element + ')');
        value = composition(element) / 100;
        tc_set_condition(condition, value);
        check_error()
    end
    tc_set_condition('t',1200);

    % Compute equilibrium and check for errors
    tc_compute_equilibrium;
    check_error()

    tic;
    temperatures = linspace(min_temperature, max_temperature, nb_temperature_steps);
    mu_c = zeros(1,length(temperatures));
    for i = 1:length(temperatures)
        tc_set_condition('t', temperatures(i));
        tc_compute_equilibrium;
        mu_c(i) = tc_get_value('mu(c)');
        check_error()
    end
    timing = toc;

    plot(temperatures, mu_c);
    xlabel("Temperature K");
    ylabel("Chemical potential C");
end


function [] = check_error()
    [a,b]=tc_error;
    if a~=0
        s=sprintf(' ERROR %d',a);
        disp(s);
        disp(b);
    end
    tc_reset_error;
end