from tc_python import *
import matplotlib.pyplot as plt
import numpy as np

"""
In this example, the precipitation of cementite during tempering of an Fe-Mn-C steel is simulated
considering three interface conditions: the usual ortho-equilibrium (OE) condition; paraequilibrium (PE)
condition; and a smooth transition from paraequilibrium to ortho-equilibrium condition (PE-OE). The
simulation results are compared with the experimental data from Miyamoto et al. [2007Miy].

Reference:
[2007Miy] G. Miyamoto, J. Oh, K. Hono, T. Furuhara, T. Maki, Effect of partitioning of Mn and Si on the
            growth kinetics of cementite in tempered Fe–0.6 mass% C martensite. Acta Mater. 55, 5027–5038 (2007)
"""

matrix_phase = "BCC_A2"
precipitate_phase = "CEMENTITE"
simplified = [GrowthRateModel.SIMPLIFIED, 1e6]
pe_automatic = [GrowthRateModel.PE_AUTOMATIC, 1e6]
para_equilibrium = [GrowthRateModel.PARA_EQ, 5]
temperature = 923.15
time_reduction_factor = 1

dataMeanRad_Fe06C = np.array([
    [2.93801150304007e+001, 4.56826235381642e+001],
    [1.20292275830692e+003, 1.86084218793426e+002],
    [3.65721147953065e+003, 2.70106645709637e+002],
    [1.82606234815132e+004, 3.46273364839596e+002],
    [8.96296517009815e+004, 6.38232996130685e+002],
    [2.63332947710959e+005, 8.10426610918323e+002],
])

dataMeanRad_Fe06C2Si = np.array([
    [2.98870405167112e+001, 2.23124735096120e+001],
    [1.22367802776784e+003, 7.72621528006772e+001],
    [3.65721147953065e+003, 1.22218494423446e+002],
    [1.85756928891421e+004, 1.98955150586902e+002],
    [8.66149517430179e+004, 2.97186478360141e+002],
])

dataMeanRad_Fe06C2Mn = np.array([
    [3.04027125122986e+001, 3.00039493211703e+001],
    [1.22367802776784e+003, 5.02625841577904e+001],
    [3.59518012159929e+003, 6.44360085851472e+001],
    [8.81094090141570e+004, 1.38381756279517e+002],
    [2.63332947710959e+005, 2.29612671013563e+002],
    [6.19407166565721e+005, 2.80629149518238e+002],
])

dataMeadRad_Fe06C1Mn = np.array([
     [2.98870405167112e+001, 1.43772673462123e+001],
     [1.22367802776784e+003, 6.14301736159690e+001],
     [3.53420090117442e+003, 6.63096542582554e+001],
     [1.85756928891421e+004, 9.90491202271334e+001],
     [8.96296517009815e+004, 1.65926981407700e+002],
     [2.63332947710959e+005, 2.27429330568957e+002],
     [6.08901165512859e+005, 3.23872030430307e+002],
])

data1_volFracFe2Mn06C = np.array([
    [32.29, 0.031361*0.75],
    [1099.8, 0.068784*0.75],
    [1.827e+04, 0.116361*0.75],
    [8.356e+04, 0.12531*0.75],
    [2.451e+05, 0.129768*0.75],
    [5.986e+05, 0.122394*0.75],
])

data2_volFracFe2Mn06C = np.array([
    [32.29, 0.0521*0.75],
    [32.29, 0.02265*0.75],
    [1099.8, 0.092578*0.75],
    [1099.8, 0.04746*0.75],
    [1.827e+04, 0.1242*0.75],
    [1.827e+04, 0.1043*0.75],
    [8.356e+04, 0.1318*0.75],
    [8.356e+04, 0.1166*0.75],
    [2.451e+05, 0.13359*0.75],
    [2.451e+05, 0.12715*0.75],
    [5.986e+05, 0.13125*0.75],
    [5.986e+05, 0.11367*0.75],
])


with TCPython():
    system = (SetUp()
              .set_cache_folder(os.path.basename(__file__) + "_cache")
              .select_thermodynamic_and_kinetic_databases_with_elements("FEDEMO", "MFEDEMO", ["Fe", "Mn", "C"])
              .select_phase(matrix_phase)
              .select_phase(precipitate_phase)
              .get_system())

    prisma_setup = (system.with_isothermal_precipitation_calculation()
                    .set_composition_unit(CompositionUnit.MASS_PERCENT)
                    .set_composition("Mn", 1.96)
                    .set_composition("C", 0.61)
                    .set_temperature(temperature)
                    )

    precip = (PrecipitatePhase(precipitate_phase).set_nucleation_at_grain_boundaries().with_growth_rate_model(simplified[0]))

    matrix = (MatrixPhase(matrix_phase).with_grain_growth_model(FixedGrainSize().set_grain_aspect_ratio(100.0)
                                        .fixed_grain_size(1e-7))
                                        .set_mobility_adjustment("all", 0.008, -70000.0))

    simplified_calc = prisma_setup.set_simulation_time(simplified[1]*time_reduction_factor).with_matrix_phase(matrix
                                            .add_precipitate_phase(precip)).calculate()

    precip.with_growth_rate_model(para_equilibrium[0])
    para_equilibrium_calc = prisma_setup.set_simulation_time(para_equilibrium[1]*time_reduction_factor).calculate()

    precip.with_growth_rate_model(pe_automatic[0])
    pe_automatic_calc = prisma_setup.set_simulation_time(pe_automatic[1]*time_reduction_factor).calculate()

    t_simple, mean_r_simple = simplified_calc.get_mean_radius_of(precipitate_phase)
    t_para, mean_r_para = para_equilibrium_calc.get_mean_radius_of(precipitate_phase)
    t_auto, mean_r_auto = pe_automatic_calc.get_mean_radius_of(precipitate_phase)

    t_mole_simple, mole_simple = simplified_calc.get_precipitate_composition_in_mole_fraction_of(precipitate_phase, "Mn")
    t_mole_para, mole_para = para_equilibrium_calc.get_precipitate_composition_in_mole_fraction_of(precipitate_phase, "Mn")
    t_mole_auto, mole_auto = pe_automatic_calc.get_precipitate_composition_in_mole_fraction_of(precipitate_phase, "Mn")

    fig, ax = plt.subplots()
    ax.loglog(t_simple, np.array(mean_r_simple)*1e9, "b-")
    ax.loglog(t_para, np.array(mean_r_para)*1e9, "r-")
    ax.loglog(t_auto, np.array(mean_r_auto)*1e9, "g-")
    plt.legend(["Simplfied", "Para-Equilibrium", "PE-Automatic"], loc=2)
    fig.suptitle("Mean radius of CEMENTITE comparing growth rate models", fontsize=11,
                 fontweight="bold")
    ax.set_xlabel("Time [s]")
    ax.set_ylabel("Length [nm]")
    x, y = dataMeanRad_Fe06C2Mn.T
    plt.scatter(x, y, marker="^", c="#ff0000")
    plt.xlim([1e-12, 1e6])
    plt.ylim([1e-1, 1e3])
    plt.tight_layout()
    plt.show()

    fig2, ax2 = plt.subplots()
    ax2.semilogx(t_mole_simple, mole_simple, "b-")
    ax2.semilogx(t_mole_para, mole_para, "r-")
    ax2.semilogx(t_mole_auto, mole_auto, "g-")
    plt.legend(["Simplified", "Para-Equilibrium", "PE-Automatic"], loc=1)
    fig2.suptitle("Mole faction Mn in CEMENTITE comparing growth rate models", fontsize=11,
                 fontweight="bold")
    ax2.set_xlabel("Time [s]")
    ax2.set_ylabel("Precipitate Composition [Mole fraction]")
    x, y = data1_volFracFe2Mn06C.T
    plt.scatter(x, y, marker="o", c="#ff0000")
    x, y = data2_volFracFe2Mn06C.T
    plt.scatter(x, y, marker="x", c="#ff0000")
    ax2.errorbar(32.39, 0.75*(0.02265+(0.0521-0.02265)/2), yerr=0.75*(0.0521-0.02265)/2, linewidth=1.0, c="#ff0000")
    ax2.errorbar(1099.8, 0.75 * (0.04746 + (0.092578 - 0.04746) / 2), yerr=0.75 * (0.092578 - 0.04746) / 2, linewidth=1.0, c="#ff0000")
    ax2.errorbar(1.827e4, 0.75*(0.1043+(0.1242-0.1043)/2), yerr=0.75*(0.1242-0.1043)/2, linewidth=1.0, c="#ff0000")
    ax2.errorbar(8.356e4, 0.75*(0.1166+(0.1318-0.1166)/2), yerr=0.75*(0.1318-0.1166)/2, linewidth=1.0, c="#ff0000")
    ax2.errorbar(2.451e5, 0.75*(0.12715+(0.13359-0.12715)/2), yerr=0.75*(0.13359-0.12715)/2, linewidth=1.0, c="#ff0000")
    ax2.errorbar(5.986e5, 0.75*(0.11367+(0.13125-0.11367)/2), yerr=0.75*(0.13125-0.11367)/2, linewidth=1.0, c="#ff0000")
    plt.xlim([1e-2, 1e6])
    plt.ylim([0, 0.14])
    plt.tight_layout()
    plt.show()