from tc_python import *
import matplotlib.pyplot as plt
import numpy as np

"""
At four temperature points (673 K, 703 K, 733 K, and 763 K), this isothermal calculation employs 
four Precipitation Calculators, each 30 K apart. It aims to investigate the mean radius of an 
Al-0.12Sc system with an FCC_A1 matrix phase and AL3SC precipitate phase featuring bulk nucleation sites. 
Notably, the user defines the interfacial energy function, utilizing an error function for a smooth 
transition of interfacial energy, transitioning from 0.065 J/m2 to 0.085 J/m2 based on particle radii, 
distinguishing between values below and above 1e-8m and 5e-8m, respectively.

A dataset based on Iwamura and Miura [2004Iwa] data is compared with the calculated results.

Reference:
[2004Iwa] Iwamura, S, and Y Miura. 2004. “Loss in Coherency and Coarsening Behavior of Al3Sc
            Precipitates.” Acta Materialia 52 (3): 591–600
"""

matrix_phase = "FCC_A1"
precipitate_phase = "AL3SC"
temperature_673 = [673, "c-"]
temperature_703 = [703, "g-"]
temperature_733 = [733, "b-"]
temperature_763 = [763, "r-"]
interfacial_energy = "0.075+0.011*erf((r-3e-8)/1e-8)".__str__()


# data at 673K to compare with the plots
symbol673 = np.array([
    [1000, 11.02],
    [10000,	10.66],
    [100000, 13.96],
    [300000, 16.90],
    [400000, 16.54],
    [800000, 19.11],
    [1000000, 20.58],
])

# data at 703K to compare with the plots
symbol703 = np.array([
    [1000, 18.01],
    [10000, 22.41],
    [100000, 22.41],
    [500000, 31.60],
    [1000000, 41.15],
])

# data at 733K to compare with the plots
symbol733 = np.array([
    [1000, 19.84],
    [10000, 23.88],
    [45000, 27.56],
    [70000, 29.40],
    [100000, 32.34],
    [200000, 35.28],
    [300000, 44.46],
    [400000, 47.77],
    [500000, 54.75],
    [800000, 66.51],
    [1000000, 75.70],
])

# data at 763K to compare with the plots
symbol763 = np.array([
    [1000, 27.19],
    [10000, 29.76],
    [30000, 40.05],
    [100000, 54.38],
    [200000, 69.45],
    [350000, 84.51],
    [700000, 103.99],
    [1000000, 121.26],
])


with TCPython():
    system = (SetUp()
              .set_cache_folder(os.path.basename(__file__) + "_cache")
              .select_thermodynamic_and_kinetic_databases_with_elements("ALDEMO", "MALDEMO", ["Al", "Sc"])
              .select_phase(matrix_phase)
              .select_phase(precipitate_phase)
              .get_system())

    prisma_setup = (system.with_isothermal_precipitation_calculation()
                    .set_composition_unit(CompositionUnit.MOLE_PERCENT)
                    .set_composition("Sc", 0.12)
                    .set_simulation_time(1e6)
                    .with_matrix_phase((MatrixPhase(matrix_phase)
                                        .add_precipitate_phase((PrecipitatePhase(precipitate_phase)
                                                                .set_nucleation_in_bulk()
                                                                .set_interfacial_energy(interfacial_energy)
                                                                ))
                                        ))
                    )

    # set temperatures (673K, 703K, 733K, 763K), then calculate and get mean radius for each setting
    time_673, mean_radius_673 = prisma_setup.set_temperature(temperature_673[0]).calculate().get_mean_radius_of(precipitate_phase)
    time_703, mean_radius_703 = prisma_setup.set_temperature(temperature_703[0]).calculate().get_mean_radius_of(precipitate_phase)
    time_733, mean_radius_733 = prisma_setup.set_temperature(temperature_733[0]).calculate().get_mean_radius_of(precipitate_phase)
    time_763, mean_radius_763 = prisma_setup.set_temperature(temperature_763[0]).calculate().get_mean_radius_of(precipitate_phase)

    fig, ax = plt.subplots()
    plt.plot(time_673, np.array(mean_radius_673) * 1e9, temperature_673[1])
    plt.plot(time_703, np.array(mean_radius_703) * 1e9, temperature_703[1])
    plt.plot(time_733, np.array(mean_radius_733) * 1e9, temperature_733[1])
    plt.plot(time_763, np.array(mean_radius_763) * 1e9, temperature_763[1])
    x, y = symbol673.T
    plt.scatter(x, y, marker="o", c="#00ffff")
    x, y = symbol703.T
    plt.scatter(x, y, marker="o", c="#8fce00")
    x, y = symbol733.T
    plt.scatter(x, y, marker="s", c="#0000FF")
    x, y = symbol763.T
    plt.scatter(x, y, marker="s", c="#f44336")
    plt.legend(["T 673K", "T 703K", "T 733K", "T 763K"], loc=1)
    fig.suptitle("Mean radius of AL3SC in Bulk at T: 673K, 703K, 733K, 763K", fontsize=11, fontweight="bold")
    ax.set_xlabel("Time [s]")
    ax.set_ylabel("Length [nm]")
    plt.tight_layout()
    plt.show()
