from tc_python import *
import numpy as np
import matplotlib.pyplot as plt

"""
In this example, three different simulations are performed: 
 - Steady-state 
 - Transient Single track
 - Transient Multilayer 

Both Transient_Single track and Transient_Multilayer use Transient with heat source from Steady-state model to compute 
time-dependent temperature distribution in the given geometry. The primary difference between this example and the 
previous examples (AM_01 to AM_03) is that, in this case, the material properties data is retrieved from the Scheil 
Calculator whereas in the previous examples the material properties are taken from the preinstalled material library.
Furthermore, this example simulates scanning of two layers of powder where the second layer is spread on the top of 
the first layer when scanning of the first layer is completed.
"""

with TCPython() as start:
    start.set_cache_folder(os.path.basename(__file__) + "_cache")

    composition = {"Al": 6.0, "V": 4.0}
    dependent_element = "Ti"
    database = "TCTI5"

    elements = list(composition.keys())
    system = (start.select_database_and_elements(database, [dependent_element] + elements)
              .get_system())

    scheil_options = (ScheilOptions().calculate_from_start_temperature()
                      .calculate_to_temperature_below_solidus()
                      .enable_evaporation_property_calculation())

    scheil_calculator = (system.with_scheil_calculation()
                         .with_options(scheil_options)
                         .set_start_temperature(5000.0)
                         .set_composition_unit(CompositionUnit.MASS_PERCENT))

    for element in composition:
        scheil_calculator.set_composition(element, composition[element])

    scheil_result = scheil_calculator.calculate()

    mp = MaterialProperties.from_scheil_result(scheil_result)

    heat_source = (HeatSource.gaussian()
                   .set_power(100.0)
                   .set_absorptivity(40.0)
                   .set_beam_radius(100.0e-6)
                   .set_scanning_speed(600e-3))

    calc_ss = (start.with_additive_manufacturing().with_steady_state_calculation()
               .with_material_properties(mp)
               .with_numerical_options(NumericalOptions().set_number_of_cores(4))
               .enable_fluid_flow_marangoni()
               .set_layer_thickness(55.0e-6)
               .with_heat_source(heat_source))

    result_ss = calc_ss.calculate()
    plotter, mesh = result_ss.get_pyvista_plotter(view_buttons=False)
    contour_mesh = mesh.contour(
        isosurfaces=np.array([mp.get_solidification_temperature(), mp.get_liquidus_temperature()]))
    plotter.add_mesh(contour_mesh)

    scale_flow_field_vectors = 1e-5
    plotter.add_arrows(mesh.points, mesh.active_vectors, mag=scale_flow_field_vectors,
                       scalar_bar_args={"title": "Velocity*{}".format(scale_flow_field_vectors)})
    plotter.camera.focal_point = (0.0, 0.0, 0.0)
    plotter.camera.zoom(10.0)
    plotter.show(interactive_update=True)

    probe_single_track_1 = ProbeCoordinate(1.3e-3, 0.5e-3, 2.055e-3)
    probe_single_track_2 = ProbeCoordinate(1.7e-3, 0.5e-3, 2.055e-3)
    calc_transient_ss = (start.with_additive_manufacturing().with_transient_with_steady_state_calculation()
                         .with_material_properties(mp)
                         .with_numerical_options(NumericalOptions().set_number_of_cores(4))
                         .with_heat_source(heat_source)
                         .enable_fluid_flow_marangoni()
                         .set_height(2.0e-3)
                         .set_width(1.0e-3)
                         .set_length(3.0e-3)
                         .set_layer_thickness(55e-6)
                         .with_scanning_strategy(ScanningStrategy().single_track().set_margin(0.75e-3))
                         .add_probe(probe_single_track_1)
                         .add_probe(probe_single_track_2))
    result_transient_ss = calc_transient_ss.calculate()
    probe_1_time, probe_1_temperature = result_transient_ss.get_temperatures_at_probe(probe_single_track_1)
    probe_2_time, probe_2_temperature = result_transient_ss.get_temperatures_at_probe(probe_single_track_2)

    fig2, axs = plt.subplots(1, 1)
    axs.plot(probe_1_time, probe_1_temperature, marker=".", label="Probe 1")
    axs.plot(probe_2_time, probe_2_temperature, marker=".", label="Probe 2")
    axs.set_xlabel("Time [s]")
    axs.set_ylabel("Temperature [K]")
    plt.legend(loc="best")
    plt.title("Probes single track - Ti64")
    plt.show(block=False)


    def update_plot(plotter, mesh):
        mesh.clip(normal="y", inplace=True, invert=False)
        plotter.add_mesh(mesh, clim=[296.15, mp.get_evaporation_temperature()], name="mymesh")


    plotter, mesh = result_transient_ss.get_pyvista_plotter(update_plot_callback=update_plot)
    update_plot(plotter, mesh)
    plotter.show(interactive_update=True)

    calc_transient_ss.remove_all_probes()
    probe_dual_layer_1 = ProbeCoordinate(1.25e-3, 1.25e-3, 2.055e-3)
    probe_dual_layer_2 = ProbeCoordinate(1.25e-3, 1.25e-3, 2.11e-3)

    (calc_transient_ss.set_height(2.0e-3)
     .set_width(2.5e-3)
     .set_length(2.5e-3)
     .add_probe(probe_dual_layer_1)
     .add_probe(probe_dual_layer_2)
     .with_scanning_strategy(ScanningStrategy().bi_directional()
                             .set_margin(0.75e-3)
                             .set_hatch_spacing(0.165e-3)
                             .set_lift_time(0.0)
                             .set_number_of_layers(2)
                             .set_powder_fill_time(2.0)
                             .set_angle(0.0)))

    result_transient_2_layer = calc_transient_ss.calculate()
    probe_dual_layer_1_time, probe_dual_layer_1_temperature = result_transient_2_layer.get_temperatures_at_probe(
        probe_dual_layer_1)
    probe_dual_layer_2_time, probe_dual_layer_2_temperature = result_transient_2_layer.get_temperatures_at_probe(
        probe_dual_layer_2)


    def update_plot(plotter, mesh):
        z_layer = mesh.slice(normal="z", origin=[0.0, 0.0, mesh.bounds[5]])
        z_contour_mesh = z_layer.contour(isosurfaces=np.linspace(300, 600, 5))
        if z_contour_mesh.points.size > 0:
            plotter.add_mesh(z_contour_mesh, render_lines_as_tubes=True, line_width=5.0,
                         clim=[296.15, mp.get_evaporation_temperature()], name="z iso-contours")
        plotter.add_mesh(mesh, show_edges=False, clim=[296.15, mp.get_evaporation_temperature()], name="surface")


    fig3, axs = plt.subplots(1, 1)
    axs.plot(probe_dual_layer_1_time, probe_dual_layer_1_temperature, marker=".", label="Probe 1")
    axs.plot(probe_dual_layer_2_time, probe_dual_layer_2_temperature, marker=".", label="Probe 2")
    axs.set_xlabel("Time [s]")
    axs.set_ylabel("Temperature [K]")
    plt.legend(loc="best")
    plt.title("Probes dual layer - Ti64")
    plt.show(block=False)

    plotter, mesh = result_transient_2_layer.get_pyvista_plotter(update_plot_callback=update_plot)
    update_plot(plotter, mesh)
    plotter.show()
