import numpy as np
from matplotlib import pyplot as plt
from scipy.interpolate import interp1d

from tc_python import *

"""
Calculates the variation of the 80% solid temperature using Scheil-solidification for random compositions
within a specification range.
The alloy system Al-Si-Cu is used as an example.
"""

num_samples = 40  # should be higher for good statistics (for example N=100)
dependent_element = "Al"
database = "ALDEMO"
lower_spec_limit = {"Si": 8.0, "Cu": 2.0}  # in wt-%
upper_spec_limit = {"Si": 11.0, "Cu": 4.0}  # in wt-%
required_solid_frac = 0.8  # the temperature for which the solid fraction will be evaluated
start_temp = 650  # in degree Celsius
final_solid_frac = 0.85  # finishing condition of the calculation (needs be higher than the required fraction)
min_content = 1e-3  # smallest element content accepted in the distribution (in wt-%)


def get_alloy_composition_distribution(lower_spec_limit, upper_spec_limit, num_samples):
    """Calculate a set of compositions from the specification range assuming a multivariate normal distribution."""
    num_dep_elements = len(lower_spec_limit)
    elements = []
    mean_composition = np.zeros(num_dep_elements)
    assumed_two_sigma_interval = np.zeros(num_dep_elements)
    for index, element in enumerate(lower_spec_limit):
        elements.append(element)
        mean_composition[index] = (lower_spec_limit[element] + upper_spec_limit[element]) / 2
        assumed_two_sigma_interval[index] = (upper_spec_limit[element] - lower_spec_limit[element]) / 2

    # build the covariance matrix (assuming a diagonal covariance with given 2-sigma confidence intervals)
    covariance_matrix = np.zeros((num_dep_elements, num_dep_elements))
    for index in np.arange(0, num_dep_elements):
        covariance_matrix[index][index] = (assumed_two_sigma_interval[index] / 2) ** 2

    composition_samples = np.random.multivariate_normal(mean_composition, covariance_matrix, num_samples)

    return composition_samples, elements


composition_samples, elements = get_alloy_composition_distribution(lower_spec_limit, upper_spec_limit, num_samples)
required_T_for_solid_frac = np.zeros(num_samples)

fig, (ax1, ax2) = plt.subplots(2, 1)
with TCPython() as session:
    system = (session.
              select_database_and_elements(database, [dependent_element] + elements).
              get_system())
    calculator = system.with_scheil_calculation()
    (calculator.
     with_options(ScheilOptions().terminate_on_fraction_of_liquid_phase(1 - final_solid_frac)).
     set_start_temperature(start_temp + 273.15).
     set_composition_unit(CompositionUnit.MASS_PERCENT))

    for index, composition in enumerate(composition_samples):
        for content, element in zip(composition, elements):
            # prevent negative concentrations generated by the distribution
            if content < min_content:  # in wt-%
                content = min_content
            calculator.set_composition(element, content)

        result = calculator.calculate()

        solid_frac, temp = result.get_values_of(ScheilQuantity.mole_fraction_of_all_solid_phases(),
                                                ScheilQuantity.temperature())
        scheil_curve = interp1d(solid_frac, temp)
        required_T_for_solid_frac[index] = scheil_curve(required_solid_frac)

        # plot the results
        ax1.plot(np.array(temp) - 273.15, solid_frac)

ax1.set_xlabel("Temperature [\N{DEGREE SIGN} C]")
ax1.set_ylabel("Solid phase fraction [-]")
ax1.set_title("Variation of the solidification curves")

ax2.hist(np.array(required_T_for_solid_frac) - 273.15)
ax2.set_xlabel("Temperature with solid fraction of {} [\N{DEGREE SIGN} C]".format(required_solid_frac))
ax2.set_ylabel("Frequency of occurrence [-]")

fig.set_size_inches(5, 7)
plt.show()

print("The used random composition set is:\n {}".format(composition_samples))
