import numpy as np

from tc_python import *
import matplotlib.pyplot as plt

"""
In this example, the precipitation of cementite during tempering of a Fe-Cr-C steel is simulated
considering two interface conditions: one is the usual ortho-equilibrium condition; the other is the para-
equilibrium condition. The simulation results are compared with the experimental data from Sakuma et
al. [1980Sak].

Reference:
[1980Sak] T. Sakuma, N. Watanabe, T. Nishizawa, The Effect of Alloying Element on the Coarsening
            Behavior of Cementite Particles in Ferrite. Trans. Japan Inst. Met. 21, 159–168 (1980).
"""

matrix_phase = "BCC_A2"
precipitate_phase = "CEMENTITE"
simplified = [GrowthRateModel.SIMPLIFIED, 600*3600]
para_equilibrium = [GrowthRateModel.PARA_EQ, 20*3600]

sakuma_data = np.array([
    [2.777778e-4*pow((6.48854961832061e+000), 3), (3.96036862714077e-002 * 1e3)],
    [2.777778e-4*pow((1.41221374045802e+001), 3), (3.95641340030851e-002 * 1e3)],
    [2.777778e-4*pow((3.20610687022901e+001), 3), (8.61033105248586e-002 * 1e3)],
    [2.777778e-4*pow((4.69465648854962e+001), 3), (9.92386188347902e-002 * 1e3)],
    [2.777778e-4*pow((6.90839694656489e+001), 3), (1.14667958707432e-001 * 1e3)],
    [2.777778e-4*pow((1.01908396946565e+002), 3), (1.25378712969189e-001 * 1e3)],
    [2.777778e-4*pow((1.18702290076336e+002), 3), (1.35395324921884e-001 * 1e3)],
])


with TCPython():
    system = (SetUp()
              .set_cache_folder(os.path.basename(__file__) + "_cache")
              .select_thermodynamic_and_kinetic_databases_with_elements("FEDEMO", "MFEDEMO", ["Fe", "Cr", "C"])
              .select_phase(matrix_phase)
              .select_phase(precipitate_phase)
              .get_system())

    prisma_setup = (system.with_isothermal_precipitation_calculation()
                    .set_composition_unit(CompositionUnit.MASS_PERCENT)
                    .set_composition("Cr", 0.95)
                    .set_composition("C", 1.065)
                    .set_temperature(773))

    precip = (PrecipitatePhase(precipitate_phase).set_nucleation_at_grain_boundaries()
              .with_growth_rate_model(simplified[0]))

    matrix = (MatrixPhase(matrix_phase).add_precipitate_phase(precip).with_grain_growth_model(FixedGrainSize()
                        .set_grain_aspect_ratio(100.0).fixed_grain_size(1e-7)))

    simplified_calc = (prisma_setup.set_simulation_time(simplified[1]).with_matrix_phase(matrix)).calculate()

    precip.with_growth_rate_model(para_equilibrium[0])
    para_equilibrium_calc = (prisma_setup.set_simulation_time(para_equilibrium[1])
                             ).calculate()

    time_simple, mean_r_simple = simplified_calc.get_mean_radius_of(precipitate_phase)
    time_para, mean_r_para = para_equilibrium_calc.get_mean_radius_of(precipitate_phase)

    fig, ax = plt.subplots()
    ax.loglog(np.array(time_simple) / 3600, np.array(mean_r_simple) / 1e-9, "b-")
    ax.loglog(np.array(time_para) / 3600, np.array(mean_r_para) / 1e-9, "r-")
    plt.legend(["Simplified", "Para-Equilibrium"], loc=1)
    fig.suptitle("Mean radius of CEMENTITE in Grain boundaries", fontsize=11, fontweight="bold")
    ax.set_xlabel("Time [h]")
    ax.set_ylabel("Length [nm]")
    x, y = sakuma_data.T
    plt.scatter(x, y, marker="o", c="#ffa500")
    plt.xlim([1e-10, 1e4])
    plt.ylim([1e-1, 1e4])
    plt.show()
