from tc_python import *
import matplotlib.pyplot as plt
import numpy as np

"""
This example simulates the kinetics of precipitation of AL3ZR_D023 from an FCC_A1 solution phase. Three different precipitate
growth models are compared, Simplified, General and Advanced. See online help for more information about the growth models. 

A dataset based on Knipling et al. [2008Kni] data is compared with the calculated results.

Reference:
[2008Kni] K. E. Knipling, D. C. Dunand, D. N. Seidman, Precipitation evolution in Al–Zr and Al–Zr–Ti alloys
            during isothermal aging at 375–425°C. Acta Mater. 56, 114–127 (2008).
"""

with TCPython():
    matrix_phase = "FCC_A1"
    precipitate_phase = "AL3ZR_D023"

    system = (SetUp().set_cache_folder(os.path.basename(__file__) + "_cache")
               .select_thermodynamic_and_kinetic_databases_with_elements("ALDEMO", "MALDEMO", ["Al", "Zr"])
               .select_phase(matrix_phase)
               .select_phase(precipitate_phase)
               .get_system())

    # Create a PrecipitatePhase object before the prisma isothermal calculation object.
    # This in order to allow for changing growth model on the precipitate object later
    precipite = PrecipitatePhase(precipitate_phase).with_growth_rate_model(GrowthRateModel.SIMPLIFIED)

    prisma_setup = (system.with_isothermal_precipitation_calculation()
               .set_composition_unit(CompositionUnit.MOLE_PERCENT)
               .set_composition("Zr", 0.2)
               .set_temperature(425.0 + 273.15)
               .set_simulation_time(400 * 60 * 60)
               .with_matrix_phase(MatrixPhase(matrix_phase).add_precipitate_phase(precipite)))

    simplified_result = prisma_setup.calculate()
    t_simple, mean_r_simple = simplified_result.get_mean_radius_of(precipitate_phase)

    precipite.with_growth_rate_model(GrowthRateModel.GENERAL)
    general_result = prisma_setup.calculate()
    t_general, mean_r_general = general_result.get_mean_radius_of(precipitate_phase)

    precipite.with_growth_rate_model(GrowthRateModel.ADVANCED)
    advanced_result = prisma_setup.calculate()
    t_advanced, mean_r_advanced = advanced_result.get_mean_radius_of(precipitate_phase)

    fig, ax = plt.subplots()
    fig.suptitle("Mean radius of AL3ZR_D023 comparing different precipitate growth models", fontsize=9, fontweight="bold")
    ax.set_xlabel("Time [h]")
    ax.set_ylabel("Mean radius [nm]")
    ax.plot(np.array(t_simple)/3600, np.array(mean_r_simple)*1e9, "g-", label="Simplified")
    ax.plot(np.array(t_general)/3600, np.array(mean_r_general)*1e9, "r-", label="General")
    ax.plot(np.array(t_advanced)/3600, np.array(mean_r_advanced)*1e9, "b-", label="Advanced")

    # Al-0.2Zr (at%)  425 C 400 hours Experiments from 2008 Knipling
    ax.errorbar(400.0,  5.1, yerr=1.7, marker="s", linewidth=4.0, capsize=7.0, label="2008 Knipling")
    ax.errorbar(400.0,  6.7, yerr=1.7, marker="o", linestyle="None", capsize=7.0, label="2008 Knipling")

    ax.legend()
    plt.show()
