"""
This example demonstrates how to call the Freeze-in Temperature Equilibrium Property Model using TC-Python.

The Freeze-in Temperature Equilibrium Property Model calculates the equilibrium material state at the specified
"freeze-in" temperature, and evaluates material properties at the specified system temperature. The assumption used for
this model is that diffusion and phase transformations are negligible for a material when below the freeze-in
temperature and therefore, the material state is kept constant for system temperatures below the freeze-in temperature.

If you do not have access to TCAL9, this example can be used by changing the variable tdb to ALDEMO or other
thermodynamic database of interest, and modifying the dependent_element and key-value pairs in the 'composition' dict()
to be compatible with the selected database.

This example also demonstrates how to collect and plot all output results of any Property Model using a dictionary.

For additional information on this Property Model, reference the description in the Graphical Mode. Alternatively, when
in Thermo-Calc, press F1 to open the Help menu and search for this Model by name, i.e., "freeze-in equilibrium Property
Model"
"""
from tc_python import *
import numpy as np
import matplotlib.pyplot as plt


temp = 400  # in degrees C
freeze_in_temp = 100  # in degrees C
list_of_temps = np.linspace(25, 550, 100)  # in degrees C
tdb = "TCAL9"  # tbd is an abbreviation for Thermodynamic DataBase
dependent_element = "Al"
composition = {'Mg': 0.675,
               'Si': 0.40
               }  # in wt-%


with TCPython() as session:
    system = (session
              .set_cache_folder(os.path.basename(__file__) + "_cache")
              .select_database_and_elements(tdb, [dependent_element] + list(composition.keys()))
              .without_default_phases()
              .select_phase('FCC_A1')
              .select_phase('MG2SI_C1')
              .get_system())

    # print the available Property Models and their names
    print("Available Property Models in the default Property Model directory: {}".format(session.get_property_models()))

    # initialize the freeze-in equilibrium Property Model and system default values
    calc = system.with_property_model_calculation('FreezeInEquilibriumPythonModel')

    (calc
     .set_temperature(temp + 273.15)  # if unchanged, the temperature is set to 1000 K by default
     .set_composition_unit(CompositionUnit.MASS_PERCENT))

    for element in composition:
        calc.set_composition(element, composition[element])

    # collect and print available arguments of the Property Model, including argument descriptions and default values
    print("Available arguments:")
    modelArgs = calc.get_arguments()
    for argument in modelArgs:
        print("Property Model argument: {}".format(argument))
        print("Description of '{}': {}".format(argument, calc.get_argument_description(argument)))
        print("Default value of '{}': {}".format(argument, calc.get_argument_default(argument)))

    # for this example, only the freeze-in temperature must be specified
    calc.set_argument('Freeze-in-temperature', freeze_in_temp + 273.15)

    # run one calculation at the default temperature to see result quantities and their descriptions
    result = calc.calculate()
    print("Available result quantities:".format(result.get_result_quantities()))
    resultQuants = result.get_result_quantities()
    for quantity in resultQuants:
        print("Description of '{}': {}".format(quantity,
                                               result.get_result_quantity_description(quantity)))

    # initialize dictionary of arrays for each result quantity
    resultData = dict()
    arraylen = len(list_of_temps)
    for quantity in resultQuants:
        resultData[quantity] = np.zeros(arraylen)

    # populate dictionary of arrays with results from calculations
    for i, temp in enumerate(list_of_temps):
        result = calc.set_temperature(temp + 273.15).calculate()
        for quantity in resultQuants:
            resultData[quantity][i] = result.get_value_of(quantity)

    # plot the values of each calculated result quantity vs list_of_temps
    for i, quantity in enumerate(resultQuants):
        plt.figure(i)
        plt.plot(list_of_temps, resultData[quantity])
        plt.xlabel("Temperature [C]")
        plt.ylabel(quantity)
        plt.interactive(False)
    plt.show()

