import numpy as np

from tc_python import *
import matplotlib.pyplot as plt

"""
This example simulates the kinetics of precipitation of gamma primate (γ’) phase from gamma (γ) phase.
The simulation results can be compared with experimental data collected from Sudbrack et al.
[2008Sud].

Reference:
[2008Sud] C. K. Sudbrack, T. D. Ziebell, R. D. Noebe, D. N. Seidman, Effects of a tungsten addition on the
            morphological evolution, spatial correlations and temporal evolution of a model Ni–Al–Cr superalloy. 
            Acta Mater. 56, 448–463 (2008).
"""

temperature = 800
simulation_time = 1000000

precipitate_phase = "FCC_L12#2"
matrix_phase = "DIS_FCC_A1"

# data for number density to compare with the plots
sym_num_density = np.array([
    [4*3600, 29e20],
    [4*3600, 24e20],
    [4*3600, 34e20],
    [16*3600, 6.8e20],
    [16*3600, 6.8e20],
    [16*3600, 5.7e20],
    [16*3600, 7.9e20],
    [64*3600, 2.1e20],
    [64*3600, 2.1e20],
    [64*3600, 1.6e20],
    [64*3600, 2.6e20],
    [264*3600, 0.7e20],
    [264*3600, 0.7e20],
    [264*3600, 0.57e20],
    [264*3600, 0.83e20],
])

# data for mean radius to compare with the plots (uncomment to see all the points plotted)
sym_mean_radius = np.array([
    [4*3600, 31.1*1e-9],
    # [4*3600, 28.4*1e-9],
    # [4*3600, 33.8*1e-9],
    # [16*3600, 58*1e-9],
    [16*3600, 58*1e-9],
    # [16*3600, 53*1e-9],
    # [16*3600, 63*1e-9],
    # [64*3600, 88*1e-9],
    [64*3600, 88*1e-9],
    # [64*3600, 83*1e-9],
    # [64*3600, 93*1e-9],
    # [264*3600, 141*1e-9],
    [264*3600, 141*1e-9],
    # [264*3600, 156*1e-9],
    # [264*3600, 126*1e-9],
])

# data for volume fraction to compare with the plots (uncomment to see all the points plotted)
sym_vol_fact = np.array([
    [4*3600, 14e-2],
    # [4*3600, 9e-2],
    # [4*3600, 19e-2],
    # [16*3600, 19.7e-2],
    [16*3600, 19.7e-2],
    # [16*3600, 18.4e-2],
    # [16*3600, 21.0e-2],
    # [64*3600, 19e-2],
    [64*3600, 19e-2],
    # [64*3600, 16e-2],
    # [64*3600, 22e-2],
    # [264*3600, 19.2e-2],
    [264*3600, 19.2e-2],
    # [264*3600, 17.9e-2],
    # [264*3600, 20.5e-2],
])


with TCPython():
    system = (SetUp()
              .set_cache_folder(os.path.basename(__file__) + "_cache")
              .select_thermodynamic_and_kinetic_databases_with_elements("NIDEMO", "MNIDEMO", ["Ni", "Al", "Cr"])
              .select_phase(matrix_phase)
              .select_phase(precipitate_phase)
              .get_system())

    precipitate = (PrecipitatePhase(precipitate_phase)
                   .set_interfacial_energy(0.012)
                   )

    prisma_setup = (system.with_isothermal_precipitation_calculation()
                    .set_composition_unit(CompositionUnit.MOLE_PERCENT)
                    .set_composition("Al", 9.8)
                    .set_composition("Cr", 8.3)
                    .set_temperature(temperature + 273.15)
                    .set_simulation_time(simulation_time)
                    .with_matrix_phase(MatrixPhase(matrix_phase).add_precipitate_phase(precipitate)))

    result = prisma_setup.calculate()

    time_1, mean_radius_1 = result.get_mean_radius_of(precipitate_phase)
    time_2, volume_2 = result.get_volume_fraction_of(precipitate_phase)
    time_3, num_density_3 = result.get_number_density_of(precipitate_phase)

    fig, ax = plt.subplots()
    ax.loglog(time_1, mean_radius_1)
    plt.plot(time_1, mean_radius_1, "b-")
    plt.legend(["Mean radius of FCC_L12#2(Bulk)"], loc=1)
    fig.suptitle("Mean radius of FCC_L12#2 in DIS_FCC_A1", fontsize=11, fontweight="bold")
    ax.set_xlabel("Time [s]")
    ax.set_ylabel("Length [m]")
    x, y = sym_mean_radius.T
    plt.scatter(x, y, marker="D", c="#474747")
    plt.xlim(1e-2, 1e6)
    plt.ylim(1e-10, 1e-6)
    plt.tight_layout()

    fig, ax1 = plt.subplots()
    plt.loglog(time_2, volume_2, "b-")
    plt.legend(["Volume of FCC_L12#2 (Bulk)"], loc=1)
    fig.suptitle("Volume of FCC_L12#2 in DIS_FCC_A1", fontsize=11, fontweight="bold")
    ax1.set_xlabel("Time [s]")
    ax1.set_ylabel("Volume Fraction")
    x, y = sym_vol_fact.T
    plt.scatter(x, y, marker="D", c="#474747")
    plt.xlim(1e-2, 1e6)
    plt.ylim(1e-2, 1)
    ax1.errorbar(4*3600, 14e-2, yerr=10e-2, marker="D", linewidth=1.0, capsize=7.0)
    ax1.errorbar(16*3600, 19.7e-2, yerr=2.6e-2, marker="D", linewidth=1.0, capsize=7.0)
    ax1.errorbar(64 * 3600, 19e-2, yerr=6e-2, marker="D", linewidth=1.0, capsize=7.0)
    ax1.errorbar(264 * 3600, 19.2e-2, yerr=2.6e-2, marker="D", linewidth=1.0, capsize=7.0)
    plt.tight_layout()

    fig, ax = plt.subplots()
    ax.loglog(time_3, num_density_3)
    plt.plot(time_3, num_density_3, "b-")
    plt.legend(["Number density of FCC_L12#2 (Bulk)"], loc=1)
    fig.suptitle("Number density of FCC_L12#2 in DIS_FCC_A1", fontsize=11, fontweight="bold")
    ax.set_xlabel("Time [s]")
    ax.set_ylabel("Number density [m-3]")
    x, y = sym_num_density.T
    plt.scatter(x, y, marker="s", c="#474747")
    plt.xlim(1e-2, 1e6)
    plt.ylim(1e10, 1e26)
    plt.tight_layout()
    plt.show()

