from tc_python import *
import matplotlib.pyplot as plt

"""
The transient model solves for the heat equation in the entire 3D domain, including the melt pool, and is therefore, 
computationally expensive to solve. The fluid flow inside the melt pool due to the Marangoni effect is not included 
in this example. The double ellipsoidal, or the so-called Goldak heat source model, is used to predict melt pool size 
and temperature distribution during single track scanning. The parameters for the double ellipsoidal heat source are 
computed using optimization in a steady-state case for the given process parameters (power and scanning speed) and the 
melt pool size reported in the paper by Grange et al. [2021Gra].

[2021Gra] D. Grange, A. Queva, G. Guillemot, M. Bellet, J.-D. Bartout, C. Colin, Effect of processing parameters during 
the laser beam melting of Inconel 738: Comparison between simulated and experimental melt pool shape. 
J. Mater. Process. Technol. 289, 116897 (2021).
"""

with TCPython() as start:
    start.set_cache_folder(os.path.basename(__file__) + "_cache")
    mp = MaterialProperties.from_library("IN738LC").set_smoothing_for_all_properties(Smoothing.LARGE)

    probe_1 = ProbeCoordinate(0.5e-3, 0.25e-3, 1.005e-3)
    probe_2 = ProbeCoordinate(0.5e-3, 0.29e-3, 1.005e-3)

    calc = (start.with_additive_manufacturing().with_transient_calculation()
            .with_numerical_options(NumericalOptions().set_number_of_cores(4))
            .with_material_properties(mp)
            .set_height(0.95e-3)
            .set_width(0.5e-3)
            .set_length(2.0e-3)
            .set_layer_thickness(55.0e-6)
            .with_heat_source(HeatSource.double_ellipsoidal()
                              .set_power(230.0)
                              .set_absorptivity(88.06)
                              .set_ar(97.3e-6)
                              .set_af(99.04e-6)
                              .set_b(41.67e-6)
                              .set_c(181.11e-6)
                              .set_scanning_speed(960.0e-3))
            .disable_fluid_flow_marangoni()
            .with_scanning_strategy(ScanningStrategy().single_track().set_margin(0.25e-3))
            .add_probe(probe_1)
            .add_probe(probe_2))

    result = calc.calculate()

    probe_1_time, probe_1_temperature = result.get_temperatures_at_probe(probe_1)
    probe_2_time, probe_2_temperature = result.get_temperatures_at_probe(probe_2)

    fig2, axs = plt.subplots(1, 1)
    axs.plot(probe_1_time, probe_1_temperature, marker=".", label="Probe 1")
    axs.plot(probe_2_time, probe_2_temperature, marker=".", label="Probe 2")
    axs.set_xlabel("Time [s]")
    axs.set_ylabel("Temperature [K]")
    plt.legend(loc="best")
    plt.show(block=False)

    # plot update function that will be called every time the slider with simulation time is dragged in the plot windows
    def update_plot(plotter, mesh):
        mesh.clip(normal="y", inplace=True, invert=False)
        plotter.add_mesh(mesh, show_edges=False, clim=[296.15, 1.2 * mp.get_evaporation_temperature()], name="mymesh")


    plotter, mesh = result.get_pyvista_plotter(update_plot_callback=update_plot)
    update_plot(plotter, mesh)
    plotter.show()
