from tc_python import *
import matplotlib.pyplot as plt

"""
This example simulates the kinetics of precipitation of gamma prime (γ’) phase from gamma (γ) phase in
Ni-8Al-8Cr and Ni-10Al-10Cr at.% alloys during continuous cooling. The simulation results can be
compared with experimental results from Rojhirunsakool et al. [2013Roj].

Reference:
[2013Roj] T. Rojhirunsakool, S. Meher, J. Y. Hwang, S. Nag, J. Tiley, R. Banerjee, Influence of composition
            on monomodal versus multimodal γ′ precipitation in Ni–Al–Cr alloys. J. Mater. Sci. 48, 825–831 (2013).
"""

initial_temperature = 1150
simulation_time = 3300
lower_temperature = 380

precipitate_phase = "FCC_L12#2"
matrix_phase = "DIS_FCC_A1"


# static helper method to get the results for the alloys
def get_results(results):
    time_1, mean_radius_1 = results.get_mean_radius_of(precipitate_phase)
    for i in time_1:
        radius, size_distribution = results.get_size_distribution_for_radius_of(precipitate_phase, i)
    return time_1, mean_radius_1, radius, size_distribution


with TCPython():
    system = (SetUp()
              .set_cache_folder(os.path.basename(__file__) + "_cache")
              .select_thermodynamic_and_kinetic_databases_with_elements("NIDEMO", "MNIDEMO", ["Ni", "Al", "Cr"])
              .select_phase(matrix_phase)
              .select_phase(precipitate_phase)
              .get_system())

    setup = (system.with_non_isothermal_precipitation_calculation()
             .set_composition_unit(CompositionUnit.MOLE_PERCENT)
             .set_composition("Al", 8.0)
             .set_composition("Cr", 8.0)
             .with_temperature_profile(TemperatureProfile()
                                       .add_time_temperature(0, initial_temperature)
                                       .add_time_temperature(simulation_time, lower_temperature))
             .set_simulation_time(simulation_time)
             .with_matrix_phase((MatrixPhase(matrix_phase)
                                 .set_mobility_adjustment("all", 5.0, 0.0)
                                 .add_precipitate_phase((PrecipitatePhase(precipitate_phase)
                                                         .set_interfacial_energy(0.023)
                                                         )))))

    result_8 = setup.calculate()

    result_10 = setup.set_composition("Al", 10.0).set_composition("Cr", 10.0).calculate()

    # use a static method to get the results for both alloys
    t8, m8, r8, s8 = get_results(result_8)
    t10, m10, r10, s10 = get_results(result_10)

    fig1, ax1 = plt.subplots()
    ax1.semilogy(t8, m8, "g-")
    ax1.set_title(r"Mean radius of FCC_L12#2 precipitate for Ni8Al-8Cr (at%)", fontsize=13)
    ax1.set_xlabel("Time [s]")
    ax1.set_ylabel("Length [m]")
    ax1.legend(["Mean radius of FCC_L12#2"], loc=1)

    fig2, ax2 = plt.subplots()
    ax2.semilogy(t10, m10, "g-")
    ax2.set_title(r"Mean radius of FCC_L12#2 precipitate for Ni10Al-10Cr (at%)", fontsize=13)
    ax2.set_xlabel("Time [s]")
    ax2.set_ylabel("Length [m]")
    ax2.legend(["Mean radius of FCC_L12#2"], loc=1)

    fig4, ax4 = plt.subplots()
    ax4.set_title(r"Size distribution of FCC_L12#2 precipitate", fontsize=13)
    ax4.set_xlabel("Length [m]")
    ax4.set_ylabel("Size distribution [m-4]")
    ax4.loglog(r8, s8)
    ax4.legend(["Size distribution - 8 mol%"], loc=1)

    fig5, ax5 = plt.subplots()
    ax5.set_title(r"Size distribution of FCC_L12#2 precipitate", fontsize=13)
    ax5.set_xlabel("Length [m]")
    ax5.set_ylabel("Size distribution [m-4]")
    ax5.loglog(r10, s10)
    plt.legend(["Size distribution - 10 mol%"], loc=1)
    plt.tight_layout()
    plt.show()

    # (Optional) Uncomment plot to compare size distributions for alloys Ni8Al-8Cr and Ni8Al-8Cr
    
    # fig3, ax3 = plt.subplots()
    # ax3.set_title(r"Comparing Ni8Al-8Cr and Ni8Al-8Cr size distributions", fontsize=13)
    # ax3.set_xlabel("Length [m]")
    # ax3.set_ylabel("Size distribution [m-4]")
    # ax3.loglog(r8, s8, "r-", r10, s10, "b-.")
    # ax3.legend(["8 mol%", "10 mol%"], loc=1)
    # plt.tight_layout()
    # plt.show()
